<?php
/**
 * Plugin Name: Randevu Eklentisi
 * Plugin URI: https://psikiyatrirehberi.com.tr/doktor-randevu-eklentisi-wordpress-icin-ucretsiz-premium-ozellikli-ve-tam-kapsamli-cozum
 * Description: Psikologlar, psikiyatristler ve tüm sağlık profesyonelleri için tasarlanmış modern randevu yönetim sistemi.
 * Version: 1.0.0
 * Author: Fatih ÖZ
 * Author URI: https://psikiyatrirehberi.com.tr/doktor-randevu-eklentisi-wordpress-icin-ucretsiz-premium-ozellikli-ve-tam-kapsamli-cozum
 * Text Domain: appointment-booking
 * Domain Path: /languages
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * License: Proprietary
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// ============================================
// LICENSE PROTECTION - DO NOT MODIFY
// ============================================
define('_ABL_D1', 'cHNpa2l5YXRyaXJlaGJlcmk=');
define('_ABL_D2', 'Y29tLnRy');
define('_ABL_HASH', 'f9ed0b3dc956');

function _abl_get_domain() {
    return base64_decode(_ABL_D1) . '.' . base64_decode(_ABL_D2);
}

function _abl_check_integrity() {
    $h = substr(md5(_ABL_D1 . _ABL_D2 . 'apb_integrity_2024'), 0, 12);
    return $h === _ABL_HASH;
}

function appointment_booking_verify_license() {
    // Integrity check - no HTTP needed
    if (!_abl_check_integrity()) {
        return false;
    }
    
    // Check cache first (24 saat)
    $cache = get_option('_abl_cache');
    if ($cache && isset($cache['t']) && isset($cache['v'])) {
        // Cache 24 saatten yeni mi?
        if ((time() - $cache['t']) < 86400) {
            return $cache['v'] === 1;
        }
    }
    
    // HTTP check
    $domain = _abl_get_domain();
    $response = wp_remote_head('https://' . $domain . '/', array(
        'timeout' => 5,
        'sslverify' => false,
        'user-agent' => 'RandevuEklentisi/1.0'
    ));
    
    $valid = false;
    if (!is_wp_error($response)) {
        $code = wp_remote_retrieve_response_code($response);
        $valid = ($code >= 200 && $code < 400);
    }
    
    // Cache result
    update_option('_abl_cache', array('t' => time(), 'v' => $valid ? 1 : 0), false);
    
    return $valid;
}

function appointment_booking_license_notice() {
    echo '<div class="notice notice-error"><p><strong>Randevu Eklentisi:</strong> Lisans doğrulaması başarısız. Eklenti devre dışı bırakıldı. Lütfen eklenti kodlarında değişiklik yapmayınız.</p></div>';
}

/**
 * Format phone number to (5XX) XXX XX XX
 */
function appointment_format_phone($phone) {
    $phone = preg_replace('/[^0-9]/', '', $phone);
    if (strlen($phone) == 10) {
        return '(' . substr($phone, 0, 3) . ') ' . substr($phone, 3, 3) . ' ' . substr($phone, 6, 2) . ' ' . substr($phone, 8, 2);
    }
    return $phone;
}

// Admin notice - her zaman göster eğer lisans geçersizse
add_action('admin_notices', function() {
    // Integrity check önce
    if (!_abl_check_integrity()) {
        echo '<div class="notice notice-error"><p><strong>Randevu Eklentisi:</strong> Lisans doğrulaması başarısız. Eklenti devre dışı bırakıldı. Lütfen eklenti kodlarında değişiklik yapmayınız.</p></div>';
        return;
    }
    
    // Backlink guard integrity check
    if (class_exists('Appointment_Booking_Backlink_Guard')) {
        $guard = new Appointment_Booking_Backlink_Guard();
        if (!$guard->verify()) {
            echo '<div class="notice notice-error"><p><strong>Randevu Eklentisi:</strong> Lisans doğrulaması başarısız. Eklenti devre dışı bırakıldı. Lütfen eklenti kodlarında değişiklik yapmayınız.</p></div>';
            return;
        }
    }
    
    // License check
    if (!appointment_booking_verify_license()) {
        echo '<div class="notice notice-error"><p><strong>Randevu Eklentisi:</strong> Lisans doğrulaması başarısız. Eklenti devre dışı bırakıldı. Lütfen eklenti kodlarında değişiklik yapmayınız.</p></div>';
    }
});

function appointment_booking_is_licensed() {
    // Integrity check her zaman (HTTP yok, hızlı)
    if (!_abl_check_integrity()) {
        return false;
    }
    // Cache varsa kullan, yoksa true dön (admin'de kontrol edilecek)
    $cache = get_option('_abl_cache');
    if ($cache && isset($cache['v'])) {
        return $cache['v'] === 1;
    }
    return true;
}

function appointment_booking_get_license_domain() {
    return _abl_get_domain();
}
// ============================================

// Define plugin constants
define('APPOINTMENT_BOOKING_VERSION', '1.0.0');
define('APPOINTMENT_BOOKING_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('APPOINTMENT_BOOKING_PLUGIN_URL', plugin_dir_url(__FILE__));
define('APPOINTMENT_BOOKING_PLUGIN_FILE', __FILE__);

// Require all class files
require_once APPOINTMENT_BOOKING_PLUGIN_DIR . 'includes/class-appointment-db.php';
require_once APPOINTMENT_BOOKING_PLUGIN_DIR . 'includes/class-appointment-calendar.php';
require_once APPOINTMENT_BOOKING_PLUGIN_DIR . 'includes/class-appointment-form.php';
require_once APPOINTMENT_BOOKING_PLUGIN_DIR . 'includes/class-appointment-admin.php';
require_once APPOINTMENT_BOOKING_PLUGIN_DIR . 'includes/class-appointment-widget.php';
require_once APPOINTMENT_BOOKING_PLUGIN_DIR . 'includes/class-appointment-backlink-guard.php';

// Activation hook
register_activation_hook(__FILE__, 'appointment_booking_activate');

function appointment_booking_activate() {
    require_once APPOINTMENT_BOOKING_PLUGIN_DIR . 'includes/class-appointment-db.php';
    $db = new Appointment_Booking_DB();
    $db->create_tables();
    
    // Force table update for existing installations
    appointment_booking_upgrade_tables();
}

/**
 * Upgrade database tables - add missing columns
 */
function appointment_booking_upgrade_tables() {
    global $wpdb;
    $blocked_table = $wpdb->prefix . 'appointment_blocked_dates';
    $appointments_table = $wpdb->prefix . 'appointments';
    
    // Check if blocked table exists first
    $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$blocked_table}'");
    if ($table_exists) {
        // Get existing columns
        $columns = $wpdb->get_col("DESCRIBE {$blocked_table}", 0);
        
        // Add start_time column if missing
        if (!in_array('start_time', $columns)) {
            $wpdb->query("ALTER TABLE {$blocked_table} ADD COLUMN start_time TIME DEFAULT NULL AFTER blocked_date");
        }
        
        // Add end_time column if missing
        if (!in_array('end_time', $columns)) {
            $wpdb->query("ALTER TABLE {$blocked_table} ADD COLUMN end_time TIME DEFAULT NULL AFTER start_time");
        }
    }
    
    // Check if appointments table exists
    $appointments_exists = $wpdb->get_var("SHOW TABLES LIKE '{$appointments_table}'");
    if ($appointments_exists) {
        // Get existing columns
        $columns = $wpdb->get_col("DESCRIBE {$appointments_table}", 0);
        
        // Add appointment_end_time column if missing
        if (!in_array('appointment_end_time', $columns)) {
            $wpdb->query("ALTER TABLE {$appointments_table} ADD COLUMN appointment_end_time TIME NULL AFTER appointment_time");
        }
    }
    
    // Set default options
    if (!get_option('appointment_booking_settings')) {
        $default_settings = array(
            'slot_duration' => 30,
            'months_ahead' => 3,
            'working_days' => array(1, 2, 3, 4, 5),
            'working_hours' => array(
                'start' => '09:00',
                'end' => '17:00'
            )
        );
        add_option('appointment_booking_settings', $default_settings);
    }
    
    flush_rewrite_rules();
}

// Deactivation hook
register_deactivation_hook(__FILE__, 'appointment_booking_deactivate');

function appointment_booking_deactivate() {
    flush_rewrite_rules();
}

// Initialize plugin
add_action('plugins_loaded', 'appointment_booking_init');

function appointment_booking_init() {
    // Check license before initializing
    if (!appointment_booking_is_licensed()) {
        // Only show admin, disable frontend
        if (!is_admin()) {
            return;
        }
    }
    
    // Load text domain
    load_plugin_textdomain('appointment-booking', false, dirname(plugin_basename(__FILE__)) . '/languages');
    
    // Check and upgrade tables if needed (runs once per version)
    $db_version = get_option('appointment_booking_db_version', '0');
    if (version_compare($db_version, '1.2', '<')) {
        appointment_booking_upgrade_tables();
        update_option('appointment_booking_db_version', '1.2');
    }
    
    // Initialize database class
    $GLOBALS['appointment_db'] = new Appointment_Booking_DB();
    
    // Initialize calendar class
    $GLOBALS['appointment_calendar'] = new Appointment_Booking_Calendar();
    
    // Initialize form class
    $GLOBALS['appointment_form'] = new Appointment_Booking_Form();
    
    // Initialize admin class (always, not just in admin)
    $GLOBALS['appointment_admin'] = new Appointment_Booking_Admin();
    
    // Widget
    add_action('widgets_init', function() {
        register_widget('Appointment_Booking_Widget');
    });
}

// Enqueue frontend assets
add_action('wp_enqueue_scripts', 'appointment_booking_enqueue_assets');

function appointment_booking_enqueue_assets() {
    // CSS - Single file for all frontend styles
    wp_enqueue_style(
        'appointment-booking-frontend',
        APPOINTMENT_BOOKING_PLUGIN_URL . 'assets/css/frontend.css',
        array(),
        APPOINTMENT_BOOKING_VERSION . '.' . time() // Cache bust
    );
    
    // JavaScript
    wp_enqueue_script(
        'appointment-booking-calendar',
        APPOINTMENT_BOOKING_PLUGIN_URL . 'assets/js/calendar.js',
        array('jquery'),
        APPOINTMENT_BOOKING_VERSION,
        true
    );
    
    wp_enqueue_script(
        'appointment-booking-form',
        APPOINTMENT_BOOKING_PLUGIN_URL . 'assets/js/form-handler.js',
        array('jquery'),
        APPOINTMENT_BOOKING_VERSION,
        true
    );
    
    // Localize script
    wp_localize_script('appointment-booking-form', 'appointmentBooking', array(
        'ajaxUrl' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('appointment_booking_nonce'),
        'strings' => array(
            'loading' => __('Yükleniyor...', 'appointment-booking'),
            'error' => __('Bir hata oluştu. Lütfen tekrar deneyin.', 'appointment-booking'),
            'success' => __('Randevunuz başarıyla kaydedildi!', 'appointment-booking'),
            'required' => __('Bu alan zorunludur.', 'appointment-booking'),
            'invalidPhone' => __('Geçersiz telefon numarası.', 'appointment-booking')
        )
    ));
}

// Register shortcode
add_shortcode('appointment_booking', 'appointment_booking_shortcode');

function appointment_booking_shortcode($atts) {
    $atts = shortcode_atts(array(
        'compact' => 'false'
    ), $atts);
    
    if (class_exists('Appointment_Booking_Form')) {
        $form = new Appointment_Booking_Form();
        return $form->render_form($atts);
    }
    
    return '';
}
