/**
 * Calendar functionality - Supports multiple forms on same page
 */

(function($) {
    'use strict';
    
    const monthNames = [
        'Ocak', 'Şubat', 'Mart', 'Nisan', 'Mayıs', 'Haziran',
        'Temmuz', 'Ağustos', 'Eylül', 'Ekim', 'Kasım', 'Aralık'
    ];
    
    // Initialize calendar for each form
    function initCalendar($wrapper) {
        let currentYear = new Date().getFullYear();
        let currentMonth = new Date().getMonth() + 1;
        let selectedDate = null;
        
        function loadCalendar(year, month) {
            const $calendar = $wrapper.find('.appointment-calendar');
            $calendar.html('<div class="calendar-loading">Yükleniyor...</div>');
            
            $.ajax({
                url: appointmentBooking.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'appointment_get_available_dates',
                    nonce: appointmentBooking.nonce,
                    year: year,
                    month: month
                },
                success: function(response) {
                    if (response.success) {
                        renderCalendar(response.data);
                    } else {
                        showError('Takvim yüklenemedi.');
                    }
                },
                error: function() {
                    showError('Bir hata oluştu. Lütfen tekrar deneyin.');
                }
            });
        }
        
        function renderCalendar(data) {
            const $calendar = $wrapper.find('.appointment-calendar');
            $calendar.empty();
            
            // Update header
            $wrapper.find('.calendar-month-year').text(monthNames[data.month - 1] + ' ' + data.year);
            
            // Add empty cells for days before month starts
            for (let i = 0; i < data.first_day_of_week; i++) {
                $calendar.append('<div class="calendar-day empty"></div>');
            }
            
            // Add days
            data.days.forEach(day => {
                let classes = ['calendar-day'];
                
                if (day.is_past) {
                    classes.push('past');
                } else if (day.is_available) {
                    classes.push('available');
                } else {
                    classes.push('unavailable');
                }
                
                if (day.is_today) {
                    classes.push('today');
                }
                
                if (selectedDate === day.date) {
                    classes.push('selected');
                }
                
                const $day = $('<div class="' + classes.join(' ') + '" data-date="' + day.date + '">' + day.day + '</div>');
                
                $calendar.append($day);
            });
        }
        
        function selectDate(date) {
            selectedDate = date;
            
            // Update UI - only in this form
            $wrapper.find('.calendar-day').removeClass('selected');
            $wrapper.find('.calendar-day[data-date="' + date + '"]').addClass('selected');
            
            // Set hidden field
            $wrapper.find('.field-appointment-date').val(date);
            
            // Format date for display
            const dateObj = new Date(date);
            const formatted = dateObj.getDate() + ' ' + monthNames[dateObj.getMonth()] + ' ' + dateObj.getFullYear();
            $wrapper.find('.selected-date-text').text(formatted);
            
            // Move to next step
            $wrapper.find('.appointment-step-2').removeClass('active');
            $wrapper.find('.appointment-step-3').addClass('active');
            updateProgressSteps(3);
            
            // Load time slots
            loadTimeSlots(date);
        }
        
        function updateProgressSteps(step) {
            $wrapper.find('.progress-step').removeClass('active completed');
            $wrapper.find('.progress-step').each(function() {
                const stepNum = parseInt($(this).data('step'));
                if (stepNum < step) {
                    $(this).addClass('completed');
                } else if (stepNum === step) {
                    $(this).addClass('active');
                }
            });
        }
        
        function loadTimeSlots(date) {
            const $slots = $wrapper.find('.appointment-time-slots');
            $slots.html('<div class="slots-loading">Yükleniyor...</div>');
            
            $.ajax({
                url: appointmentBooking.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'appointment_get_time_slots',
                    nonce: appointmentBooking.nonce,
                    date: date
                },
                success: function(response) {
                    if (response.success) {
                        renderTimeSlots(response.data.slots);
                    } else {
                        showError('Saatler yüklenemedi.');
                    }
                },
                error: function() {
                    showError('Bir hata oluştu. Lütfen tekrar deneyin.');
                }
            });
        }
        
        function renderTimeSlots(slots) {
            const $container = $wrapper.find('.appointment-time-slots');
            $container.empty();
            
            if (slots.length === 0) {
                $container.html('<div class="no-slots">Bu tarihte müsait saat bulunmamaktadır.</div>');
                return;
            }
            
            slots.forEach(slot => {
                let classes = ['time-slot'];
                
                if (!slot.is_available) {
                    classes.push('unavailable');
                }
                
                const $slot = $('<div class="' + classes.join(' ') + '" data-time="' + slot.time + '">' + slot.time + '</div>');
                $container.append($slot);
            });
        }
        
        function showError(message) {
            const $messages = $wrapper.find('.appointment-messages');
            $messages.html('<div class="appointment-message error">' + message + '</div>');
            
            setTimeout(function() {
                $messages.empty();
            }, 5000);
        }
        
        // Calendar navigation - scoped to this form
        $wrapper.find('.calendar-prev').on('click', function() {
            currentMonth--;
            if (currentMonth < 1) {
                currentMonth = 12;
                currentYear--;
            }
            loadCalendar(currentYear, currentMonth);
        });
        
        $wrapper.find('.calendar-next').on('click', function() {
            currentMonth++;
            if (currentMonth > 12) {
                currentMonth = 1;
                currentYear++;
            }
            loadCalendar(currentYear, currentMonth);
        });
        
        // Day click handler - scoped to this form
        $wrapper.on('click', '.calendar-day.available', function() {
            const date = $(this).data('date');
            selectDate(date);
        });
        
        // Listen for calendar load trigger from form handler
        $wrapper.on('loadCalendar', function() {
            loadCalendar(currentYear, currentMonth);
        });
    }
    
    // Initialize all calendars on page
    $(document).ready(function() {
        $('.appointment-booking-wrapper').each(function() {
            initCalendar($(this));
        });
    });
    
})(jQuery);
