/**
 * Form handling and validation - Supports multiple forms on same page
 */

(function($) {
    'use strict';
    
    // Initialize each form independently
    function initForm($wrapper) {
        const $form = $wrapper.find('.appointment-form');
        let currentStep = 1;
        
        function validateStep1() {
            let isValid = true;
            
            // Clear previous errors
            $wrapper.find('.appointment-field input').removeClass('error');
            
            // Validate name
            const name = $wrapper.find('.field-user-name').val().trim();
            if (name === '') {
                $wrapper.find('.field-user-name').addClass('error');
                isValid = false;
            }
            
            // Validate surname
            const surname = $wrapper.find('.field-user-surname').val().trim();
            if (surname === '') {
                $wrapper.find('.field-user-surname').addClass('error');
                isValid = false;
            }
            
            // Validate phone
            const phone = $wrapper.find('.field-user-phone').val().trim();
            if (phone === '') {
                $wrapper.find('.field-user-phone').addClass('error');
                isValid = false;
            } else if (!isValidPhone(phone)) {
                $wrapper.find('.field-user-phone').addClass('error');
                showMessage('Geçersiz telefon numarası. Lütfen geçerli bir Türkiye telefon numarası girin.', 'error');
                isValid = false;
            }
            
            if (!isValid) {
                showMessage('Lütfen tüm zorunlu alanları doldurun.', 'error');
            }
            
            return isValid;
        }
        
        function isValidPhone(phone) {
            phone = phone.replace(/[\s\-\(\)]/g, '');
            
            // Remove leading 0 if present
            if (phone.startsWith('0')) {
                phone = phone.slice(1);
            }
            
            // Remove country code if present
            if (phone.startsWith('90')) {
                phone = phone.slice(2);
            }
            
            // Turkish mobile: 5XXXXXXXXX (10 digits starting with 5)
            if (/^5[0-9]{9}$/.test(phone)) {
                return true;
            }
            
            // Turkish landline: 2XX, 3XX, 4XX area codes (10 digits)
            if (/^[234][0-9]{9}$/.test(phone)) {
                return true;
            }
            
            return false;
        }
        
        function goToStep(step) {
            $wrapper.find('.appointment-step').removeClass('active');
            $wrapper.find('.appointment-step-' + step).addClass('active');
            currentStep = step;
            
            // Update progress
            updateProgressSteps(step);
            
            // Reset submit button when going back
            if (step !== 3) {
                $wrapper.find('.field-appointment-time').val('');
                $wrapper.find('.appointment-btn-submit').prop('disabled', true);
            }
        }
        
        function updateProgressSteps(step) {
            $wrapper.find('.progress-step').removeClass('active completed');
            $wrapper.find('.progress-step').each(function() {
                const stepNum = parseInt($(this).data('step'));
                if (stepNum < step) {
                    $(this).addClass('completed');
                } else if (stepNum === step) {
                    $(this).addClass('active');
                }
            });
        }
        
        function showMessage(message, type) {
            const $messages = $wrapper.find('.appointment-messages');
            $messages.html('<div class="appointment-message ' + type + '">' + message + '</div>');
            
            setTimeout(function() {
                $messages.empty();
            }, type === 'success' ? 10000 : 5000);
        }
        
        function resetForm() {
            $form[0].reset();
            $wrapper.find('.time-slot').removeClass('selected');
            $wrapper.find('.calendar-day').removeClass('selected');
            $wrapper.find('.appointment-btn-submit').prop('disabled', true);
            goToStep(1);
        }
        
        // Ensure submit button is disabled initially
        $wrapper.find('.appointment-btn-submit').prop('disabled', true);
        
        // Next button (Step 1 -> Step 2)
        $wrapper.find('.appointment-step-1 .appointment-btn-next').on('click', function() {
            if (validateStep1()) {
                goToStep(2);
                // Trigger calendar load
                $wrapper.trigger('loadCalendar');
            }
        });
        
        // Back buttons
        $wrapper.find('.appointment-step-2 .appointment-btn-back').on('click', function() {
            goToStep(1);
        });
        
        $wrapper.find('.appointment-step-3 .appointment-btn-back').on('click', function() {
            goToStep(2);
            // Reset time selection
            $wrapper.find('.field-appointment-time').val('');
            $wrapper.find('.time-slot').removeClass('selected');
            $wrapper.find('.appointment-btn-submit').prop('disabled', true);
        });
        
        // Time slot selection
        $wrapper.on('click', '.time-slot:not(.unavailable)', function() {
            // Remove previous selection in this form only
            $wrapper.find('.time-slot').removeClass('selected');
            
            // Select this slot
            $(this).addClass('selected');
            
            // Get time value
            const time = $(this).data('time');
            $wrapper.find('.field-appointment-time').val(time);
            
            // Don't enable submit yet - wait for captcha
            checkSubmitButton();
        });
        
        // Captcha input - check if can enable submit
        $wrapper.on('input', '.captcha-input', function() {
            checkSubmitButton();
        });
        
        // Check if submit button should be enabled
        function checkSubmitButton() {
            const timeSelected = $wrapper.find('.field-appointment-time').val() !== '';
            const captchaFilled = $wrapper.find('.captcha-input').val().trim() !== '';
            
            if (timeSelected && captchaFilled) {
                $wrapper.find('.appointment-btn-submit').prop('disabled', false);
            } else {
                $wrapper.find('.appointment-btn-submit').prop('disabled', true);
            }
        }
        
        // Form submission
        $form.on('submit', function(e) {
            e.preventDefault();
            
            const $submitBtn = $wrapper.find('.appointment-btn-submit');
            const appointmentTime = $wrapper.find('.field-appointment-time').val();
            const appointmentDate = $wrapper.find('.field-appointment-date').val();
            
            // Validate time is selected
            if (!appointmentTime || appointmentTime === '') {
                showMessage('Lütfen bir saat seçin.', 'error');
                $submitBtn.prop('disabled', false).text('Randevu Al');
                return false;
            }
            
            // Validate date is selected
            if (!appointmentDate || appointmentDate === '') {
                showMessage('Lütfen bir tarih seçin.', 'error');
                $submitBtn.prop('disabled', false).text('Randevu Al');
                return false;
            }
            
            // Validate captcha is filled
            var captchaVal = $wrapper.find('.captcha-input').val();
            if (!captchaVal || captchaVal.trim() === '') {
                showMessage('Lütfen doğrulama sorusunu cevaplayın.', 'error');
                $submitBtn.prop('disabled', false).text('Randevu Al');
                return false;
            }
            
            // Disable submit button
            $submitBtn.prop('disabled', true).text('Gönderiliyor...');
            
            const formData = {
                action: 'appointment_submit',
                nonce: appointmentBooking.nonce,
                user_name: $wrapper.find('.field-user-name').val(),
                user_surname: $wrapper.find('.field-user-surname').val(),
                user_phone: $wrapper.find('.field-user-phone').val(),
                appointment_date: appointmentDate,
                appointment_time: appointmentTime,
                captcha_num1: $wrapper.find('.captcha-num1').val(),
                captcha_num2: $wrapper.find('.captcha-num2').val(),
                captcha_answer: $wrapper.find('.captcha-input').val()
            };
            
            $.ajax({
                url: appointmentBooking.ajaxUrl,
                type: 'POST',
                data: formData,
                success: function(response) {
                    if (response.success) {
                        showMessage(response.data.message, 'success');
                        resetForm();
                    } else {
                        showMessage(response.data.message, 'error');
                        $submitBtn.prop('disabled', false).text('Randevu Al');
                    }
                },
                error: function() {
                    showMessage('Bir hata oluştu. Lütfen tekrar deneyin.', 'error');
                    $submitBtn.prop('disabled', false).text('Randevu Al');
                }
            });
        });
        
        // Real-time phone formatting - 532 281 56 44 format (without leading 0)
        $wrapper.find('.field-user-phone').on('input', function() {
            let value = $(this).val().replace(/\D/g, '');
            
            // Remove leading 0 if present
            if (value.startsWith('0')) {
                value = value.slice(1);
            }
            
            // Limit to 10 digits
            if (value.length > 10) {
                value = value.slice(0, 10);
            }
            
            // Format: 532 281 56 44
            let formatted = '';
            if (value.length > 0) {
                formatted = value.slice(0, 3); // 532
            }
            if (value.length > 3) {
                formatted += ' ' + value.slice(3, 6); // 281
            }
            if (value.length > 6) {
                formatted += ' ' + value.slice(6, 8); // 56
            }
            if (value.length > 8) {
                formatted += ' ' + value.slice(8, 10); // 44
            }
            
            $(this).val(formatted);
        });
        
        // Input validation on blur
        $wrapper.find('.appointment-field input').on('blur', function() {
            $(this).toggleClass('error', $(this).val().trim() === '');
        });
    }
    
    // Initialize all forms on page
    $(document).ready(function() {
        $('.appointment-booking-wrapper').each(function() {
            initForm($(this));
        });
    });
    
})(jQuery);
