<?php
/**
 * Admin panel class
 *
 * @package Appointment_Booking
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Appointment_Booking_Admin {
    
    /**
     * @var Appointment_Booking_DB
     */
    private $db;
    
    /**
     * Format phone number to (5XX) XXX XX XX
     */
    private function format_phone($phone) {
        $phone = preg_replace('/[^0-9]/', '', $phone);
        if (strlen($phone) == 10) {
            return '(' . substr($phone, 0, 3) . ') ' . substr($phone, 3, 3) . ' ' . substr($phone, 6, 2) . ' ' . substr($phone, 8, 2);
        }
        return $phone;
    }
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->db = new Appointment_Booking_DB();
        
        add_action('admin_menu', array($this, 'register_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        add_action('admin_post_appointment_action', array($this, 'handle_appointment_action'));
        add_action('admin_post_appointment_save_working_hours', array($this, 'handle_save_working_hours'));
        add_action('admin_post_appointment_block_date', array($this, 'handle_block_date'));
        add_action('admin_post_appointment_unblock_date', array($this, 'handle_unblock_date'));
        add_action('admin_post_appointment_save_email_settings', array($this, 'handle_save_email_settings'));
        add_action('admin_post_appointment_save_custom_settings', array($this, 'handle_save_custom_settings'));
        
        // AJAX handlers
        add_action('wp_ajax_appointment_admin_update_status', array($this, 'ajax_update_status'));
        add_action('wp_ajax_appointment_admin_delete', array($this, 'ajax_delete_appointment'));
        add_action('wp_ajax_appointment_test_email', array($this, 'ajax_test_email'));
        add_action('wp_ajax_appointment_admin_add', array($this, 'ajax_admin_add_appointment'));
    }
    
    /**
     * Register admin menu
     */
    public function register_menu() {
        // Get pending appointments count for badge
        $pending_count = 0;
        if (class_exists('Appointment_Booking_DB')) {
            $db = new Appointment_Booking_DB();
            $all_appointments = $db->get_appointments(array('status' => 'pending'));
            $pending_count = count($all_appointments);
        }
        
        // Menu title with badge
        $menu_title = $pending_count > 0 
            ? sprintf(__('Randevular %s', 'appointment-booking'), '<span class="awaiting-mod">' . $pending_count . '</span>')
            : __('Randevular', 'appointment-booking');
        
        add_menu_page(
            __('Randevular', 'appointment-booking'),
            $menu_title,
            'manage_options',
            'appointment-booking',
            array($this, 'render_dashboard'),
            'dashicons-calendar-alt',
            30
        );
        
        add_submenu_page(
            'appointment-booking',
            __('Tüm Randevular', 'appointment-booking'),
            __('Tüm Randevular', 'appointment-booking'),
            'manage_options',
            'appointment-booking',
            array($this, 'render_dashboard')
        );
        
        add_submenu_page(
            'appointment-booking',
            __('Takvimim', 'appointment-booking'),
            __('Takvimim', 'appointment-booking'),
            'manage_options',
            'appointment-booking-calendar',
            array($this, 'render_calendar_page')
        );
        
        add_submenu_page(
            'appointment-booking',
            __('Çalışma Saatleri', 'appointment-booking'),
            __('Çalışma Saatleri', 'appointment-booking'),
            'manage_options',
            'appointment-booking-hours',
            array($this, 'render_working_hours_page')
        );
        
        add_submenu_page(
            'appointment-booking',
            __('Blokeli Tarihler', 'appointment-booking'),
            __('Blokeli Tarihler', 'appointment-booking'),
            'manage_options',
            'appointment-booking-blocked',
            array($this, 'render_blocked_dates_page')
        );
        
        add_submenu_page(
            'appointment-booking',
            __('Danışanlar', 'appointment-booking'),
            __('Danışanlar', 'appointment-booking'),
            'manage_options',
            'appointment-booking-clients',
            array($this, 'render_clients_page')
        );
        
        add_submenu_page(
            'appointment-booking',
            __('Ayarlar', 'appointment-booking'),
            __('Ayarlar', 'appointment-booking'),
            'manage_options',
            'appointment-booking-settings',
            array($this, 'render_settings_page')
        );
    }
    
    /**
     * Enqueue admin assets
     */
    public function enqueue_admin_assets($hook) {
        // Premium sidebar menu styling - always load
        wp_add_inline_style('wp-admin', '
            /* Premium Sidebar Menu Styling - Compact & Elegant */
            #adminmenu #toplevel_page_appointment-booking {
                position: relative;
                margin: 2px 0 !important;
                transition: all 0.2s ease !important;
            }
            
            #adminmenu #toplevel_page_appointment-booking > a {
                background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%) !important;
                border-left: 4px solid #fbbf24 !important;
                transition: all 0.2s ease !important;
            }
            
            #adminmenu #toplevel_page_appointment-booking:hover > a {
                background: linear-gradient(135deg, #d97706 0%, #b45309 100%) !important;
                border-left-width: 6px !important;
            }
            
            #adminmenu #toplevel_page_appointment-booking .wp-menu-name {
                color: #fff !important;
                font-weight: 600 !important;
                text-shadow: 0 1px 2px rgba(0,0,0,0.15) !important;
            }
            
            #adminmenu #toplevel_page_appointment-booking .wp-menu-image:before {
                color: #fff !important;
                font-size: 18px !important;
            }
            
            #adminmenu #toplevel_page_appointment-booking.wp-has-current-submenu > a,
            #adminmenu #toplevel_page_appointment-booking.current > a {
                background: linear-gradient(135deg, #d97706 0%, #b45309 100%) !important;
                border-left: 6px solid #fbbf24 !important;
            }
            
            /* Badge for pending appointments - Compact */
            #adminmenu #toplevel_page_appointment-booking .awaiting-mod {
                position: absolute;
                top: 8px;
                right: 12px;
                background: linear-gradient(135deg, #dc2626 0%, #991b1b 100%) !important;
                color: #fff !important;
                border-radius: 10px !important;
                padding: 2px 7px !important;
                font-size: 10px !important;
                font-weight: 700 !important;
                box-shadow: 0 2px 6px rgba(220, 38, 38, 0.5) !important;
                animation: pulse 2s infinite !important;
                min-width: 18px;
                text-align: center;
            }
            
            @keyframes pulse {
                0%, 100% { transform: scale(1); opacity: 1; }
                50% { transform: scale(1.05); opacity: 0.9; }
            }
            
            /* Submenu styling */
            #adminmenu #toplevel_page_appointment-booking .wp-submenu {
                background: #f8f9fa !important;
            }
            
            #adminmenu #toplevel_page_appointment-booking .wp-submenu a {
                color: #4a5568 !important;
                transition: all 0.15s !important;
            }
            
            #adminmenu #toplevel_page_appointment-booking .wp-submenu a:hover {
                color: #d97706 !important;
                background: rgba(251, 191, 36, 0.1) !important;
            }
            
            #adminmenu #toplevel_page_appointment-booking .wp-submenu .current {
                background: rgba(251, 191, 36, 0.15) !important;
                color: #d97706 !important;
                font-weight: 600 !important;
                border-left: 3px solid #f59e0b !important;
            }
            
            /* Collapsed sidebar */
            @media only screen and (max-width: 960px) {
                #adminmenu #toplevel_page_appointment-booking .awaiting-mod {
                    top: 6px;
                    right: 8px;
                    padding: 1px 5px !important;
                    font-size: 9px !important;
                }
            }
            
            /* Folded sidebar (when collapsed) */
            .folded #adminmenu #toplevel_page_appointment-booking .awaiting-mod {
                position: absolute;
                top: 4px;
                right: 4px;
                left: auto;
            }
        ');
        
        $is_our_page = (
            strpos($hook, 'appointment-booking') !== false ||
            strpos($hook, 'appointment_booking') !== false ||
            (isset($_GET['page']) && strpos($_GET['page'], 'appointment-booking') !== false)
        );
        
        if (!$is_our_page) {
            return;
        }
        
        wp_enqueue_style('dashicons');
        
        wp_enqueue_style(
            'flatpickr',
            'https://cdn.jsdelivr.net/npm/flatpickr/dist/flatpickr.min.css',
            array(),
            '4.6.13'
        );
        
        wp_enqueue_script(
            'flatpickr',
            'https://cdn.jsdelivr.net/npm/flatpickr',
            array(),
            '4.6.13',
            true
        );
        
        wp_enqueue_script(
            'flatpickr-tr',
            'https://cdn.jsdelivr.net/npm/flatpickr/dist/l10n/tr.js',
            array('flatpickr'),
            '4.6.13',
            true
        );
        
        wp_enqueue_style(
            'appointment-booking-admin',
            APPOINTMENT_BOOKING_PLUGIN_URL . 'assets/css/admin.css',
            array('dashicons'),
            APPOINTMENT_BOOKING_VERSION . '.' . time()
        );
        
        wp_enqueue_script(
            'appointment-booking-admin',
            APPOINTMENT_BOOKING_PLUGIN_URL . 'assets/js/admin.js',
            array('jquery', 'flatpickr'),
            APPOINTMENT_BOOKING_VERSION . '.' . time(),
            true
        );
        
        wp_localize_script('appointment-booking-admin', 'appointmentAdmin', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('appointment_admin_nonce')
        ));
    }
    
    /**
     * Render dashboard
     */
    public function render_dashboard() {
        if (!current_user_can('manage_options')) {
            wp_die(__('Bu sayfayı görüntüleme yetkiniz yok.', 'appointment-booking'));
        }
        
        $status = isset($_GET['status']) ? sanitize_text_field($_GET['status']) : '';
        $date_from = isset($_GET['date_from']) ? sanitize_text_field($_GET['date_from']) : '';
        $date_to = isset($_GET['date_to']) ? sanitize_text_field($_GET['date_to']) : '';
        $search = isset($_GET['search']) ? sanitize_text_field($_GET['search']) : '';
        $current_page = isset($_GET['paged']) ? max(1, absint($_GET['paged'])) : 1;
        $per_page = 20;
        
        $filters = array();
        if ($status) $filters['status'] = $status;
        if ($date_from) $filters['date_from'] = $date_from;
        if ($date_to) $filters['date_to'] = $date_to;
        if ($search) $filters['search'] = $search;
        
        $total_items = $this->db->count_appointments($filters);
        $total_pages = ceil($total_items / $per_page);
        
        $filters['per_page'] = $per_page;
        $filters['page'] = $current_page;
        
        $appointments = $this->db->get_appointments($filters);
        
        $all_appointments = $this->db->get_appointments();
        $stats = array(
            'total' => count($all_appointments),
            'pending' => count(array_filter($all_appointments, function($a) { return $a->status === 'pending'; })),
            'confirmed' => count(array_filter($all_appointments, function($a) { return $a->status === 'confirmed'; })),
            'rejected' => count(array_filter($all_appointments, function($a) { return $a->status === 'rejected'; }))
        );
        
        $pagination = array(
            'current_page' => $current_page,
            'total_pages' => $total_pages,
            'total_items' => $total_items,
            'per_page' => $per_page
        );
        
        include APPOINTMENT_BOOKING_PLUGIN_DIR . 'templates/admin-dashboard.php';
    }

    /**
     * Render settings page - Shortcodes, Usage & Disclaimer
     */
    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            wp_die(__('Bu sayfayı görüntüleme yetkiniz yok.', 'appointment-booking'));
        }
        
        $email_saved = isset($_GET['email_saved']);
        $custom_saved = isset($_GET['custom_saved']);
        ?>
        <style>
        .ab-wrap { margin: 20px 20px 20px 0; font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; }
        .ab-header { display: flex; align-items: center; gap: 12px; margin-bottom: 24px; padding-bottom: 16px; border-bottom: 1px solid #e5e7eb; }
        .ab-header h1 { margin: 0; font-size: 24px; font-weight: 600; color: #111827; display: flex; align-items: center; gap: 10px; }
        .ab-header .dashicons { font-size: 28px; width: 28px; height: 28px; color: #3b82f6; }
        .ab-card { background: #fff; border: 1px solid #e5e7eb; border-radius: 12px; margin-bottom: 24px; overflow: hidden; }
        .ab-card-header { padding: 16px 20px; border-bottom: 1px solid #e5e7eb; display: flex; align-items: center; gap: 10px; background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%); }
        .ab-card-header h2 { margin: 0; font-size: 16px; font-weight: 600; color: #111827; }
        .ab-card-header .dashicons { color: #3b82f6; }
        .ab-card-body { padding: 24px; }
        .shortcode-grid { display: grid; grid-template-columns: repeat(2, 1fr); gap: 20px; }
        @media (max-width: 768px) { .shortcode-grid { grid-template-columns: 1fr; } }
        .shortcode-box { background: linear-gradient(135deg, #eff6ff 0%, #dbeafe 100%); border: 2px solid #93c5fd; border-radius: 12px; padding: 20px; display: flex; flex-direction: column; }
        .shortcode-box h3 { margin: 0 0 10px; font-size: 16px; font-weight: 600; color: #1e40af; }
        .shortcode-box p { margin: 0 0 16px; font-size: 13px; color: #3b82f6; line-height: 1.5; flex: 1; }
        .shortcode-code { display: flex; align-items: center; gap: 10px; background: #fff; border: 1px solid #93c5fd; border-radius: 8px; padding: 12px 14px; }
        .shortcode-code code { flex: 1; font-size: 13px; font-weight: 600; color: #1e40af; background: none; font-family: 'Consolas', 'Monaco', monospace; word-break: break-all; }
        .btn-copy { padding: 8px 14px; background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%); color: #fff; border: none; border-radius: 6px; font-size: 12px; font-weight: 500; cursor: pointer; transition: all 0.2s; white-space: nowrap; flex-shrink: 0; }
        .btn-copy:hover { transform: scale(1.05); box-shadow: 0 2px 8px rgba(59,130,246,0.4); }
        .btn-copy.copied { background: linear-gradient(135deg, #10b981 0%, #059669 100%); }
        .usage-list { list-style: none; padding: 0; margin: 0; }
        .usage-list li { display: flex; gap: 12px; padding: 14px 0; border-bottom: 1px solid #f3f4f6; }
        .usage-list li:last-child { border-bottom: none; }
        .usage-number { width: 28px; height: 28px; background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%); color: #fff; border-radius: 50%; display: flex; align-items: center; justify-content: center; font-size: 13px; font-weight: 600; flex-shrink: 0; }
        .usage-content h4 { margin: 0 0 4px; font-size: 14px; font-weight: 600; color: #111827; }
        .usage-content p { margin: 0; font-size: 13px; color: #6b7280; line-height: 1.5; }
        .disclaimer-box { background: linear-gradient(135deg, #fef3c7 0%, #fde68a 100%); border: 2px solid #f59e0b; border-radius: 12px; padding: 24px; }
        .disclaimer-header { display: flex; align-items: center; gap: 12px; margin-bottom: 16px; }
        .disclaimer-icon { width: 48px; height: 48px; background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); border-radius: 12px; display: flex; align-items: center; justify-content: center; font-size: 24px; }
        .disclaimer-header h3 { margin: 0; font-size: 18px; font-weight: 600; color: #92400e; }
        .disclaimer-content { font-size: 14px; color: #78350f; line-height: 1.7; }
        .disclaimer-content p { margin: 0 0 12px; }
        .disclaimer-content ul { margin: 12px 0; padding-left: 20px; }
        .disclaimer-content li { margin-bottom: 8px; }
        .disclaimer-footer { margin-top: 16px; padding-top: 16px; border-top: 1px solid #fbbf24; font-size: 13px; color: #92400e; font-style: italic; }
        .plugin-info { display: flex; align-items: flex-start; gap: 16px; padding: 20px; background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%); border-radius: 12px; border: 1px solid #e5e7eb; }
        .plugin-logo { width: 56px; height: 56px; background: linear-gradient(135deg, #3b82f6 0%, #8b5cf6 100%); border-radius: 14px; display: flex; align-items: center; justify-content: center; color: #fff; font-size: 28px; flex-shrink: 0; }
        .plugin-details h4 { margin: 0 0 6px; font-size: 17px; font-weight: 700; color: #111827; }
        .plugin-details .plugin-desc { margin: 0 0 8px; font-size: 13px; color: #374151; line-height: 1.5; }
        .plugin-details .plugin-meta { margin: 0; font-size: 12px; color: #6b7280; }
        .plugin-details .plugin-meta a { color: #3b82f6; text-decoration: none; }
        .plugin-details .plugin-meta a:hover { text-decoration: underline; }
        
        /* Email Settings */
        .email-section { margin-bottom: 24px; }
        .email-toggle-row { display: flex; align-items: center; gap: 16px; padding: 16px 20px; background: linear-gradient(135deg, #ecfdf5 0%, #d1fae5 100%); border: 2px solid #6ee7b7; border-radius: 12px; }
        .toggle-switch { position: relative; width: 52px; height: 28px; flex-shrink: 0; }
        .toggle-switch.small { width: 44px; height: 24px; }
        .toggle-switch input { opacity: 0; width: 0; height: 0; }
        .toggle-slider { position: absolute; cursor: pointer; top: 0; left: 0; right: 0; bottom: 0; background: #d1d5db; transition: 0.3s; border-radius: 28px; }
        .toggle-slider:before { position: absolute; content: ""; height: 20px; width: 20px; left: 4px; bottom: 4px; background: white; transition: 0.3s; border-radius: 50%; box-shadow: 0 2px 4px rgba(0,0,0,0.2); }
        .toggle-switch.small .toggle-slider:before { height: 16px; width: 16px; }
        .toggle-switch input:checked + .toggle-slider { background: linear-gradient(135deg, #10b981 0%, #059669 100%); }
        .toggle-switch input:checked + .toggle-slider:before { transform: translateX(24px); }
        .toggle-switch.small input:checked + .toggle-slider:before { transform: translateX(20px); }
        .toggle-label strong { display: block; font-size: 15px; color: #065f46; margin-bottom: 2px; }
        .toggle-label span { font-size: 13px; color: #047857; }
        
        .email-settings-grid { display: grid; grid-template-columns: repeat(3, 1fr); gap: 16px; margin-bottom: 24px; }
        @media (max-width: 900px) { .email-settings-grid { grid-template-columns: 1fr 1fr; } }
        @media (max-width: 600px) { .email-settings-grid { grid-template-columns: 1fr; } }
        
        .email-form-group { display: flex; flex-direction: column; gap: 6px; }
        .email-form-group label { font-size: 13px; font-weight: 600; color: #374151; display: flex; align-items: center; gap: 6px; }
        .email-form-group label .dashicons { font-size: 16px; width: 16px; height: 16px; color: #6b7280; }
        .email-form-group input, .email-form-group select { padding: 12px 14px; border: 2px solid #e5e7eb; border-radius: 10px; font-size: 14px; background: #f9fafb; transition: all 0.2s; }
        .email-form-group input:focus, .email-form-group select:focus { outline: none; border-color: #10b981; background: #fff; box-shadow: 0 0 0 4px rgba(16,185,129,0.1); }
        .email-form-group .field-hint { font-size: 11px; color: #9ca3af; }
        
        .smtp-section { background: #f8fafc; border: 2px solid #e5e7eb; border-radius: 12px; padding: 20px; margin-bottom: 24px; }
        .smtp-header { display: flex; align-items: flex-start; gap: 14px; margin-bottom: 20px; padding-bottom: 16px; border-bottom: 1px solid #e5e7eb; }
        .smtp-header-text strong { display: block; font-size: 14px; color: #374151; margin-bottom: 2px; }
        .smtp-header-text span { font-size: 12px; color: #6b7280; line-height: 1.4; }
        
        .smtp-settings-grid { display: grid; grid-template-columns: 2fr 1fr 1fr; gap: 16px; }
        .smtp-settings-grid .email-form-group:nth-child(4),
        .smtp-settings-grid .email-form-group:nth-child(5) { grid-column: span 1; }
        @media (max-width: 768px) { .smtp-settings-grid { grid-template-columns: 1fr 1fr; } }
        @media (max-width: 500px) { .smtp-settings-grid { grid-template-columns: 1fr; } }
        
        .email-actions { display: flex; gap: 12px; flex-wrap: wrap; }
        .btn-save-email { display: inline-flex; align-items: center; gap: 8px; padding: 12px 24px; background: linear-gradient(135deg, #10b981 0%, #059669 100%); color: #fff; border: none; border-radius: 10px; font-size: 14px; font-weight: 600; cursor: pointer; transition: all 0.2s; box-shadow: 0 4px 12px rgba(16,185,129,0.3); }
        .btn-save-email:hover { transform: translateY(-2px); box-shadow: 0 6px 16px rgba(16,185,129,0.4); }
        .btn-test-email { display: inline-flex; align-items: center; gap: 8px; padding: 12px 24px; background: #fff; color: #374151; border: 2px solid #e5e7eb; border-radius: 10px; font-size: 14px; font-weight: 600; cursor: pointer; transition: all 0.2s; }
        .btn-test-email:hover { border-color: #10b981; color: #059669; background: #ecfdf5; }
        
        /* Customization Settings */
        .custom-settings-grid { display: grid; grid-template-columns: 1fr 1fr; gap: 24px; margin-bottom: 20px; }
        @media (max-width: 600px) { .custom-settings-grid { grid-template-columns: 1fr; } }
        .custom-form-group { display: flex; flex-direction: column; gap: 8px; }
        .custom-form-group label { font-size: 14px; font-weight: 600; color: #374151; display: flex; align-items: center; gap: 8px; }
        .custom-form-group label .dashicons { color: #8b5cf6; }
        .custom-form-group .field-hint { font-size: 12px; color: #9ca3af; }
        .color-picker-wrap { display: flex; gap: 10px; align-items: center; }
        .color-picker { width: 50px; height: 40px; border: none; border-radius: 8px; cursor: pointer; padding: 0; }
        .color-text { padding: 10px 14px; border: 2px solid #e5e7eb; border-radius: 8px; font-size: 14px; font-weight: 600; width: 100px; background: #f9fafb; }
        .font-select { padding: 12px 14px; border: 2px solid #e5e7eb; border-radius: 10px; font-size: 14px; background: #f9fafb; width: 100%; }
        .font-select:focus { outline: none; border-color: #8b5cf6; }
        .color-presets { display: flex; align-items: center; gap: 8px; flex-wrap: wrap; margin-bottom: 20px; }
        .presets-label { font-size: 13px; color: #6b7280; margin-right: 4px; }
        .color-preset { width: 28px; height: 28px; border-radius: 6px; border: 2px solid transparent; cursor: pointer; transition: all 0.2s; }
        .color-preset:hover { transform: scale(1.15); }
        .color-preset.active { border-color: #1f2937; box-shadow: 0 0 0 2px #fff, 0 0 0 4px #1f2937; }
        .btn-save-custom { display: inline-flex; align-items: center; gap: 8px; padding: 12px 24px; background: linear-gradient(135deg, #8b5cf6 0%, #7c3aed 100%); color: #fff; border: none; border-radius: 10px; font-size: 14px; font-weight: 600; cursor: pointer; transition: all 0.2s; box-shadow: 0 4px 12px rgba(139,92,246,0.3); }
        .btn-save-custom:hover { transform: translateY(-2px); box-shadow: 0 6px 16px rgba(139,92,246,0.4); }
        </style>
        
        <div class="wrap">
        <div class="ab-wrap">
            <?php if ($email_saved): ?>
            <div style="background: linear-gradient(135deg, #ecfdf5 0%, #d1fae5 100%); border: 2px solid #6ee7b7; color: #065f46; padding: 14px 20px; border-radius: 12px; margin-bottom: 20px; display: flex; align-items: center; gap: 10px; font-weight: 500;">
                ✓ E-posta ayarları kaydedildi.
            </div>
            <?php endif; ?>
            <?php if ($custom_saved): ?>
            <div style="background: linear-gradient(135deg, #f3e8ff 0%, #e9d5ff 100%); border: 2px solid #c084fc; color: #6b21a8; padding: 14px 20px; border-radius: 12px; margin-bottom: 20px; display: flex; align-items: center; gap: 10px; font-weight: 500;">
                ✓ Görünüm ayarları kaydedildi.
            </div>
            <?php endif; ?>
            <div class="ab-header">
                <h1><span class="dashicons dashicons-admin-settings"></span> Ayarlar</h1>
            </div>
            
            <div class="ab-card">
                <div class="ab-card-header">
                    <span class="dashicons dashicons-shortcode"></span>
                    <h2>Kısa Kodlar</h2>
                </div>
                <div class="ab-card-body">
                    <div class="shortcode-grid">
                        <div class="shortcode-box">
                            <h3>📄 Sayfa İçi Form</h3>
                            <p>Tam genişlikte randevu formu için kullanın. Sayfa veya yazı içeriğine ekleyin.</p>
                            <div class="shortcode-code">
                                <code>[appointment_booking]</code>
                                <button type="button" class="btn-copy" data-code="[appointment_booking]">Kopyala</button>
                            </div>
                        </div>
                        <div class="shortcode-box">
                            <h3>📱 Sidebar / Widget</h3>
                            <p>Kompakt versiyon. Sidebar veya dar alanlarda kullanım için idealdir.</p>
                            <div class="shortcode-code">
                                <code>[appointment_booking compact="true"]</code>
                                <button type="button" class="btn-copy" data-code='[appointment_booking compact="true"]'>Kopyala</button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="ab-card">
                <div class="ab-card-header">
                    <span class="dashicons dashicons-book"></span>
                    <h2>Kullanım Kılavuzu</h2>
                </div>
                <div class="ab-card-body">
                    <ol class="usage-list">
                        <li>
                            <span class="usage-number">1</span>
                            <div class="usage-content">
                                <h4>Çalışma Saatlerini Ayarlayın</h4>
                                <p>Sol menüden "Çalışma Saatleri" sayfasına gidin. Hangi günler çalıştığınızı ve saat aralıklarını belirleyin. Randevu süresini seçin (15, 30, 45 veya 60 dakika).</p>
                            </div>
                        </li>
                        <li>
                            <span class="usage-number">2</span>
                            <div class="usage-content">
                                <h4>Tatil/İzin Günlerini Blokla</h4>
                                <p>"Blokeli Tarihler" sayfasından müsait olmadığınız günleri veya belirli saat aralıklarını blokla. Bloklu tarihlerde randevu alınamaz.</p>
                            </div>
                        </li>
                        <li>
                            <span class="usage-number">3</span>
                            <div class="usage-content">
                                <h4>Kısa Kodu Sayfaya Ekleyin</h4>
                                <p>Yukarıdaki kısa kodlardan birini kopyalayın ve randevu formunun görünmesini istediğiniz sayfaya yapıştırın. Widget alanları için kompakt versiyonu kullanın.</p>
                            </div>
                        </li>
                        <li>
                            <span class="usage-number">4</span>
                            <div class="usage-content">
                                <h4>Randevuları Yönetin</h4>
                                <p>"Randevular" sayfasından gelen talepleri görüntüleyin. Bekleyen randevuları onaylayın veya reddedin. Filtreleme ve arama özellikleriyle kolayca yönetin.</p>
                            </div>
                        </li>
                    </ol>
                </div>
            </div>
            
            <!-- Görünüm Özelleştirme -->
            <div class="ab-card">
                <div class="ab-card-header">
                    <span class="dashicons dashicons-art" style="color: #8b5cf6;"></span>
                    <h2>Görünüm Özelleştirme</h2>
                </div>
                <div class="ab-card-body">
                    <?php 
                    $custom_settings = get_option('appointment_booking_custom', array());
                    $primary_color = isset($custom_settings['primary_color']) ? $custom_settings['primary_color'] : '#2563eb';
                    $font_family = isset($custom_settings['font_family']) ? $custom_settings['font_family'] : '';
                    $font_size = isset($custom_settings['font_size']) ? $custom_settings['font_size'] : 'normal';
                    ?>
                    
                    <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                        <input type="hidden" name="action" value="appointment_save_custom_settings">
                        <?php wp_nonce_field('appointment_custom_settings_nonce'); ?>
                        
                        <div class="custom-settings-grid">
                            <div class="custom-form-group">
                                <label><span class="dashicons dashicons-admin-appearance"></span> Ana Renk</label>
                                <div class="color-picker-wrap">
                                    <input type="color" name="primary_color" value="<?php echo esc_attr($primary_color); ?>" class="color-picker">
                                    <input type="text" value="<?php echo esc_attr($primary_color); ?>" class="color-text" readonly>
                                </div>
                                <span class="field-hint">Form başlığı, butonlar ve vurgular için kullanılır</span>
                            </div>
                            
                            <div class="custom-form-group">
                                <label><span class="dashicons dashicons-editor-textcolor"></span> Yazı Tipi</label>
                                <select name="font_family" class="font-select">
                                    <option value="" <?php selected($font_family, ''); ?>>Varsayılan (Sistem)</option>
                                    <option value="'Inter', sans-serif" <?php selected($font_family, "'Inter', sans-serif"); ?>>Inter</option>
                                    <option value="'Poppins', sans-serif" <?php selected($font_family, "'Poppins', sans-serif"); ?>>Poppins</option>
                                    <option value="'Roboto', sans-serif" <?php selected($font_family, "'Roboto', sans-serif"); ?>>Roboto</option>
                                    <option value="'Open Sans', sans-serif" <?php selected($font_family, "'Open Sans', sans-serif"); ?>>Open Sans</option>
                                    <option value="'Nunito', sans-serif" <?php selected($font_family, "'Nunito', sans-serif"); ?>>Nunito</option>
                                    <option value="'Lato', sans-serif" <?php selected($font_family, "'Lato', sans-serif"); ?>>Lato</option>
                                    <option value="'Montserrat', sans-serif" <?php selected($font_family, "'Montserrat', sans-serif"); ?>>Montserrat</option>
                                    <option value="'Raleway', sans-serif" <?php selected($font_family, "'Raleway', sans-serif"); ?>>Raleway</option>
                                    <option value="'Ubuntu', sans-serif" <?php selected($font_family, "'Ubuntu', sans-serif"); ?>>Ubuntu</option>
                                    <option value="'Playfair Display', serif" <?php selected($font_family, "'Playfair Display', serif"); ?>>Playfair Display</option>
                                    <option value="'Merriweather', serif" <?php selected($font_family, "'Merriweather', serif"); ?>>Merriweather</option>
                                    <option value="'Source Sans Pro', sans-serif" <?php selected($font_family, "'Source Sans Pro', sans-serif"); ?>>Source Sans Pro</option>
                                    <option value="'Quicksand', sans-serif" <?php selected($font_family, "'Quicksand', sans-serif"); ?>>Quicksand</option>
                                    <option value="'Josefin Sans', sans-serif" <?php selected($font_family, "'Josefin Sans', sans-serif"); ?>>Josefin Sans</option>
                                    <option value="'Work Sans', sans-serif" <?php selected($font_family, "'Work Sans', sans-serif"); ?>>Work Sans</option>
                                </select>
                            </div>
                            
                            <div class="custom-form-group">
                                <label><span class="dashicons dashicons-editor-expand"></span> Yazı Boyutu</label>
                                <select name="font_size" class="font-select">
                                    <option value="small" <?php selected($font_size, 'small'); ?>>Küçük</option>
                                    <option value="normal" <?php selected($font_size, 'normal'); ?>>Normal</option>
                                    <option value="large" <?php selected($font_size, 'large'); ?>>Büyük</option>
                                    <option value="xlarge" <?php selected($font_size, 'xlarge'); ?>>Çok Büyük</option>
                                </select>
                                <span class="field-hint">Form içindeki tüm yazıların boyutu</span>
                            </div>
                        </div>
                        
                        <div class="color-presets">
                            <span class="presets-label">Hazır Renkler:</span>
                            <button type="button" class="color-preset" data-color="#2563eb" style="background:#2563eb;" title="Mavi"></button>
                            <button type="button" class="color-preset" data-color="#059669" style="background:#059669;" title="Yeşil"></button>
                            <button type="button" class="color-preset" data-color="#dc2626" style="background:#dc2626;" title="Kırmızı"></button>
                            <button type="button" class="color-preset" data-color="#7c3aed" style="background:#7c3aed;" title="Mor"></button>
                            <button type="button" class="color-preset" data-color="#ea580c" style="background:#ea580c;" title="Turuncu"></button>
                            <button type="button" class="color-preset" data-color="#0891b2" style="background:#0891b2;" title="Turkuaz"></button>
                            <button type="button" class="color-preset" data-color="#4f46e5" style="background:#4f46e5;" title="İndigo"></button>
                            <button type="button" class="color-preset" data-color="#be185d" style="background:#be185d;" title="Pembe"></button>
                        </div>
                        
                        <button type="submit" class="btn-save-custom">
                            <span class="dashicons dashicons-saved"></span> Kaydet
                        </button>
                    </form>
                </div>
            </div>
            
            <!-- E-posta Bildirimleri -->
            <div class="ab-card">
                <div class="ab-card-header">
                    <span class="dashicons dashicons-email-alt" style="color: #10b981;"></span>
                    <h2>E-posta Bildirimleri</h2>
                </div>
                <div class="ab-card-body">
                    <?php 
                    $email_settings = get_option('appointment_booking_email_settings', array());
                    $notifications_enabled = isset($email_settings['enabled']) ? $email_settings['enabled'] : false;
                    $admin_email = isset($email_settings['admin_email']) ? $email_settings['admin_email'] : get_option('admin_email');
                    $smtp_enabled = isset($email_settings['smtp_enabled']) ? $email_settings['smtp_enabled'] : false;
                    $smtp_host = isset($email_settings['smtp_host']) ? $email_settings['smtp_host'] : '';
                    $smtp_port = isset($email_settings['smtp_port']) ? $email_settings['smtp_port'] : '587';
                    $smtp_user = isset($email_settings['smtp_user']) ? $email_settings['smtp_user'] : '';
                    $smtp_pass = isset($email_settings['smtp_pass']) ? $email_settings['smtp_pass'] : '';
                    $smtp_secure = isset($email_settings['smtp_secure']) ? $email_settings['smtp_secure'] : 'tls';
                    $from_name = isset($email_settings['from_name']) ? $email_settings['from_name'] : get_bloginfo('name');
                    $from_email = isset($email_settings['from_email']) ? $email_settings['from_email'] : get_option('admin_email');
                    ?>
                    
                    <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                        <input type="hidden" name="action" value="appointment_save_email_settings">
                        <?php wp_nonce_field('appointment_email_settings_nonce'); ?>
                        
                        <div class="email-section">
                            <div class="email-toggle-row">
                                <label class="toggle-switch">
                                    <input type="checkbox" name="notifications_enabled" value="1" <?php checked($notifications_enabled); ?>>
                                    <span class="toggle-slider"></span>
                                </label>
                                <div class="toggle-label">
                                    <strong>E-posta Bildirimlerini Aktifleştir</strong>
                                    <span>Yeni randevu geldiğinde e-posta bildirimi gönder</span>
                                </div>
                            </div>
                        </div>
                        
                        <div class="email-settings-grid" id="emailSettingsGrid">
                            <div class="email-form-group">
                                <label><span class="dashicons dashicons-admin-users"></span> Bildirim Alacak E-posta</label>
                                <input type="email" name="admin_email" value="<?php echo esc_attr($admin_email); ?>" placeholder="ornek@domain.com">
                                <span class="field-hint">Yeni randevu bildirimleri bu adrese gönderilir</span>
                            </div>
                            
                            <div class="email-form-group">
                                <label><span class="dashicons dashicons-businessperson"></span> Gönderen Adı</label>
                                <input type="text" name="from_name" value="<?php echo esc_attr($from_name); ?>" placeholder="Site Adı">
                            </div>
                            
                            <div class="email-form-group">
                                <label><span class="dashicons dashicons-email"></span> Gönderen E-posta</label>
                                <input type="email" name="from_email" value="<?php echo esc_attr($from_email); ?>" placeholder="bilgi@siteadiniz.com">
                            </div>
                        </div>
                        
                        <div class="smtp-section">
                            <div class="smtp-header">
                                <label class="toggle-switch small">
                                    <input type="checkbox" name="smtp_enabled" value="1" <?php checked($smtp_enabled); ?> id="smtpToggle">
                                    <span class="toggle-slider"></span>
                                </label>
                                <div class="smtp-header-text">
                                    <strong>SMTP Ayarları (Opsiyonel)</strong>
                                    <span>Özel SMTP sunucusu kullanmak için aktifleştirin. Kapalıysa WordPress varsayılan mail fonksiyonu kullanılır.</span>
                                </div>
                            </div>
                            
                            <div class="smtp-settings-grid" id="smtpSettingsGrid">
                                <div class="email-form-group">
                                    <label>SMTP Sunucu</label>
                                    <input type="text" name="smtp_host" value="<?php echo esc_attr($smtp_host); ?>" placeholder="smtp.gmail.com">
                                </div>
                                
                                <div class="email-form-group">
                                    <label>Port</label>
                                    <input type="number" name="smtp_port" value="<?php echo esc_attr($smtp_port); ?>" placeholder="587">
                                </div>
                                
                                <div class="email-form-group">
                                    <label>Güvenlik</label>
                                    <select name="smtp_secure">
                                        <option value="tls" <?php selected($smtp_secure, 'tls'); ?>>TLS</option>
                                        <option value="ssl" <?php selected($smtp_secure, 'ssl'); ?>>SSL</option>
                                        <option value="" <?php selected($smtp_secure, ''); ?>>Yok</option>
                                    </select>
                                </div>
                                
                                <div class="email-form-group">
                                    <label>SMTP Kullanıcı Adı</label>
                                    <input type="text" name="smtp_user" value="<?php echo esc_attr($smtp_user); ?>" placeholder="kullanici@gmail.com">
                                </div>
                                
                                <div class="email-form-group">
                                    <label>SMTP Şifre</label>
                                    <input type="password" name="smtp_pass" value="<?php echo esc_attr($smtp_pass); ?>" placeholder="••••••••">
                                </div>
                            </div>
                        </div>
                        
                        <div class="email-actions">
                            <button type="submit" class="btn-save-email">
                                <span class="dashicons dashicons-saved"></span> Kaydet
                            </button>
                            <button type="button" class="btn-test-email" id="testEmailBtn">
                                <span class="dashicons dashicons-email"></span> Test E-postası Gönder
                            </button>
                        </div>
                    </form>
                </div>
            </div>
            
            <div class="ab-card">
                <div class="ab-card-header">
                    <span class="dashicons dashicons-warning" style="color: #f59e0b;"></span>
                    <h2>Sorumluluk Reddi</h2>
                </div>
                <div class="ab-card-body">
                    <div class="disclaimer-box">
                        <div class="disclaimer-header">
                            <div class="disclaimer-icon">⚠️</div>
                            <h3>Önemli Bilgilendirme</h3>
                        </div>
                        <div class="disclaimer-content">
                            <p>Bu eklenti, randevu yönetimi için bir araç olarak sunulmaktadır. Eklentiyi kullanarak aşağıdaki koşulları kabul etmiş sayılırsınız:</p>
                            <ul>
                                <li>Bu eklentinin kullanımından doğabilecek herhangi bir doğrudan veya dolaylı zarardan eklenti geliştiricisi sorumlu tutulamaz.</li>
                                <li>Randevu sisteminin kesintisiz ve hatasız çalışacağı garanti edilmez. Teknik aksaklıklar yaşanabilir.</li>
                                <li>Kullanıcı verileri (isim, telefon vb.) sitenizin veritabanında saklanır. Bu verilerin güvenliği ve KVKK uyumluluğu site sahibinin sorumluluğundadır.</li>
                                <li>Eklenti "olduğu gibi" sunulmaktadır. Herhangi bir garanti verilmemektedir.</li>
                                <li>Randevu onayları ve iptalleri tamamen site yöneticisinin kontrolündedir.</li>
                            </ul>
                            <p>Bu eklentiyi kullanarak, yukarıdaki koşulları okuduğunuzu ve kabul ettiğinizi beyan etmiş olursunuz.</p>
                        </div>
                        <div class="disclaimer-footer">
                            © <?php echo date('Y'); ?> Randevu Eklentisi - Tüm hakları saklıdır.
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="ab-card">
                <div class="ab-card-body">
                    <div class="plugin-info">
                        <div class="plugin-logo">📅</div>
                        <div class="plugin-details">
                            <h4>Randevu Eklentisi</h4>
                            <p class="plugin-desc">Psikologlar, psikiyatristler ve tüm sağlık profesyonelleri için tasarlanmış modern randevu yönetim sistemi.</p>
                            <p class="plugin-meta">Versiyon 1.0.0 • Geliştiren: <strong>Fatih ÖZ</strong> • <a href="https://psikiyatrirehberi.com.tr/doktor-randevu-eklentisi-wordpress-icin-ucretsiz-premium-ozellikli-ve-tam-kapsamli-cozum" target="_blank">Eklenti Sitesi</a></p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            $('.btn-copy').on('click', function() {
                var code = $(this).data('code');
                var $btn = $(this);
                navigator.clipboard.writeText(code).then(function() {
                    $btn.text('Kopyalandı!').addClass('copied');
                    setTimeout(function() {
                        $btn.text('Kopyala').removeClass('copied');
                    }, 2000);
                });
            });
            
            // SMTP toggle
            function toggleSmtpSettings() {
                if ($('#smtpToggle').is(':checked')) {
                    $('#smtpSettingsGrid').slideDown(200);
                } else {
                    $('#smtpSettingsGrid').slideUp(200);
                }
            }
            toggleSmtpSettings();
            $('#smtpToggle').on('change', toggleSmtpSettings);
            
            // Color picker sync
            $('input[name="primary_color"]').on('input', function() {
                $('.color-text').val($(this).val());
                $('.color-preset').removeClass('active');
            });
            
            // Color preset buttons
            $('.color-preset').on('click', function(e) {
                e.preventDefault();
                var color = $(this).data('color');
                $('input[name="primary_color"]').val(color);
                $('.color-text').val(color);
                $('.color-preset').removeClass('active');
                $(this).addClass('active');
            });
            
            // Test email button
            $('#testEmailBtn').on('click', function() {
                var $btn = $(this);
                $btn.prop('disabled', true).html('<span class="dashicons dashicons-update spin"></span> Gönderiliyor...');
                
                $.post(ajaxurl, {
                    action: 'appointment_test_email',
                    nonce: '<?php echo wp_create_nonce("appointment_test_email_nonce"); ?>'
                }, function(response) {
                    if (response.success) {
                        $btn.html('<span class="dashicons dashicons-yes"></span> Gönderildi!').css('color', '#059669');
                    } else {
                        $btn.html('<span class="dashicons dashicons-no"></span> Hata!').css('color', '#dc2626');
                        alert(response.data.message || 'E-posta gönderilemedi.');
                    }
                    setTimeout(function() {
                        $btn.prop('disabled', false).html('<span class="dashicons dashicons-email"></span> Test E-postası Gönder').css('color', '');
                    }, 3000);
                }).fail(function() {
                    $btn.prop('disabled', false).html('<span class="dashicons dashicons-email"></span> Test E-postası Gönder');
                    alert('Bağlantı hatası.');
                });
            });
        });
        </script>
        <style>.spin { animation: spin 1s linear infinite; } @keyframes spin { 100% { transform: rotate(360deg); } }</style>
        <?php
    }

    /**
     * Render Working Hours Page
     */
    public function render_working_hours_page() {
        if (!current_user_can('manage_options')) {
            wp_die(__('Bu sayfayı görüntüleme yetkiniz yok.', 'appointment-booking'));
        }
        
        $settings = get_option('appointment_booking_settings', array());
        $days_schedule = array();
        for ($i = 0; $i <= 6; $i++) {
            $hours = $this->db->get_working_hours($i);
            $days_schedule[$i] = $hours;
        }
        
        $day_names = array(
            1 => 'Pazartesi', 
            2 => 'Salı',
            3 => 'Çarşamba',
            4 => 'Perşembe',
            5 => 'Cuma',
            6 => 'Cumartesi',
            0 => 'Pazar'
        );
        ?>
        <style>
        .ab-wrap { margin: 20px 20px 20px 0; font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; }
        .ab-header { display: flex; align-items: center; gap: 12px; margin-bottom: 24px; padding-bottom: 16px; border-bottom: 1px solid #e5e7eb; }
        .ab-header h1 { margin: 0; font-size: 24px; font-weight: 600; color: #111827; display: flex; align-items: center; gap: 10px; }
        .ab-header .dashicons { font-size: 28px; width: 28px; height: 28px; color: #3b82f6; }
        .ab-card { background: #fff; border: 1px solid #e5e7eb; border-radius: 16px; margin-bottom: 24px; box-shadow: 0 1px 3px rgba(0,0,0,0.05); }
        .ab-card-header { padding: 18px 24px; border-bottom: 1px solid #e5e7eb; display: flex; align-items: center; gap: 10px; background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%); border-radius: 16px 16px 0 0; }
        .ab-card-header h2 { margin: 0; font-size: 16px; font-weight: 600; color: #111827; }
        .ab-card-header .dashicons { color: #3b82f6; }
        .ab-card-body { padding: 24px; }
        
        /* Day rows */
        .day-row { display: grid; grid-template-columns: 40px 110px 1fr auto; align-items: center; padding: 16px 20px; margin: 0 -24px; border-bottom: 1px solid #f3f4f6; gap: 20px; transition: background 0.2s; }
        .day-row:hover { background: #f8fafc; }
        .day-row:last-of-type { border-bottom: none; }
        .day-row.inactive { opacity: 0.5; }
        
        /* Toggle switch */
        .day-toggle { position: relative; }
        .day-toggle input { width: 44px; height: 24px; cursor: pointer; appearance: none; background: #d1d5db; border-radius: 12px; transition: all 0.3s; }
        .day-toggle input:checked { background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%); }
        .day-toggle input::before { content: ''; position: absolute; top: 50%; left: 3px; transform: translateY(-50%); width: 18px; height: 18px; background: #fff; border-radius: 50%; transition: all 0.3s; box-shadow: 0 1px 3px rgba(0,0,0,0.2); }
        .day-toggle input:checked::before { left: 23px; }
        
        /* Day name */
        .day-name { font-weight: 600; color: #374151; font-size: 15px; }
        
        /* Time inputs */
        .day-times { display: flex; align-items: center; gap: 10px; }
        .time-input-wrap { position: relative; }
        .time-input { padding: 10px 14px; border: 2px solid #e5e7eb; border-radius: 10px; font-size: 14px; width: 100px; background: #f9fafb; transition: all 0.2s; font-weight: 500; }
        .time-input:focus { outline: none; border-color: #3b82f6; background: #fff; box-shadow: 0 0 0 4px rgba(59,130,246,0.1); }
        .time-separator { color: #9ca3af; font-weight: 500; }
        
        /* Duration select */
        .day-duration { }
        .duration-select { padding: 10px 14px; border: 2px solid #e5e7eb; border-radius: 10px; font-size: 13px; background: #f9fafb; cursor: pointer; min-width: 110px; font-weight: 500; color: #374151; transition: all 0.2s; }
        .duration-select:focus { outline: none; border-color: #3b82f6; background: #fff; box-shadow: 0 0 0 4px rgba(59,130,246,0.1); }
        
        /* Default duration selector */
        .default-duration { margin-top: 24px; padding: 20px 24px; background: linear-gradient(135deg, #eff6ff 0%, #dbeafe 100%); border-radius: 12px; border: 2px solid #93c5fd; margin-left: -24px; margin-right: -24px; margin-bottom: -24px; border-radius: 0 0 16px 16px; }
        .duration-info-box { display: flex; align-items: flex-start; gap: 12px; background: #fff; border: 1px solid #93c5fd; border-radius: 10px; padding: 14px 16px; margin-bottom: 16px; }
        .duration-info-box .dashicons { color: #3b82f6; font-size: 20px; width: 20px; height: 20px; flex-shrink: 0; margin-top: 2px; }
        .duration-info-box p { margin: 0; font-size: 13px; color: #1e40af; line-height: 1.5; }
        .default-duration-row { display: flex; align-items: center; gap: 16px; flex-wrap: wrap; }
        .default-duration-label { font-weight: 600; color: #1e40af; font-size: 14px; display: flex; align-items: center; gap: 8px; }
        .default-duration-label .dashicons { font-size: 18px; width: 18px; height: 18px; }
        .duration-options { display: flex; gap: 10px; flex-wrap: wrap; }
        .duration-option { padding: 10px 20px; border: 2px solid #93c5fd; border-radius: 10px; cursor: pointer; font-size: 14px; font-weight: 600; transition: all 0.2s; background: #fff; color: #1e40af; }
        .duration-option:hover { background: #dbeafe; }
        .duration-option.active { background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%); color: #fff; border-color: #3b82f6; box-shadow: 0 4px 12px rgba(59,130,246,0.3); }
        .duration-option input { display: none; }
        
        /* Save button */
        .btn-save { display: inline-flex; align-items: center; gap: 10px; padding: 14px 28px; background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%); color: #fff; border: none; border-radius: 12px; font-size: 15px; font-weight: 600; cursor: pointer; transition: all 0.2s; box-shadow: 0 4px 14px rgba(59,130,246,0.35); }
        .btn-save:hover { transform: translateY(-2px); box-shadow: 0 6px 20px rgba(59,130,246,0.45); }
        .btn-save .dashicons { font-size: 18px; width: 18px; height: 18px; }
        
        /* Notice */
        .notice-success { background: linear-gradient(135deg, #ecfdf5 0%, #d1fae5 100%); border: 2px solid #6ee7b7; color: #065f46; padding: 14px 20px; border-radius: 12px; margin-bottom: 20px; display: flex; align-items: center; gap: 10px; font-weight: 500; }
        </style>
        
        <div class="wrap">
        <?php if (isset($_GET['updated'])): ?>
        <div class="notice-success"><p>✓ Çalışma saatleri kaydedildi.</p></div>
        <?php endif; ?>
        
        <div class="ab-wrap">
            <div class="ab-header">
                <h1><span class="dashicons dashicons-clock"></span> Çalışma Saatleri</h1>
            </div>
            
            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                <input type="hidden" name="action" value="appointment_save_working_hours">
                <?php wp_nonce_field('appointment_settings_nonce'); ?>
                
                <div class="ab-card">
                    <div class="ab-card-header">
                        <span class="dashicons dashicons-calendar-alt"></span>
                        <h2>Haftalık Çalışma Planı</h2>
                    </div>
                    <div class="ab-card-body">
                        <?php foreach ($day_names as $day_num => $day_name): 
                            $is_active = !empty($days_schedule[$day_num]);
                            $start = $is_active && isset($days_schedule[$day_num][0]) ? substr($days_schedule[$day_num][0]->start_time, 0, 5) : '09:00';
                            $end = $is_active && isset($days_schedule[$day_num][0]) ? substr($days_schedule[$day_num][0]->end_time, 0, 5) : '17:00';
                            $day_duration = $is_active && isset($days_schedule[$day_num][0]) ? $days_schedule[$day_num][0]->slot_duration : null;
                        ?>
                        <div class="day-row <?php echo !$is_active ? 'inactive' : ''; ?>">
                            <div class="day-toggle">
                                <input type="checkbox" name="days[<?php echo $day_num; ?>][active]" value="1" <?php checked($is_active); ?>>
                            </div>
                            <div class="day-name"><?php echo esc_html($day_name); ?></div>
                            <div class="day-times">
                                <input type="time" name="days[<?php echo $day_num; ?>][ranges][0][start]" value="<?php echo esc_attr($start); ?>" class="time-input">
                                <span class="time-separator">—</span>
                                <input type="time" name="days[<?php echo $day_num; ?>][ranges][0][end]" value="<?php echo esc_attr($end); ?>" class="time-input">
                            </div>
                            <div class="day-duration">
                                <select name="days[<?php echo $day_num; ?>][ranges][0][duration]" class="duration-select">
                                    <option value="">⏱️ Varsayılan</option>
                                    <?php foreach ([15, 30, 45, 60] as $dur): ?>
                                    <option value="<?php echo $dur; ?>" <?php selected($day_duration, $dur); ?>><?php echo $dur; ?> dakika</option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                        <?php endforeach; ?>
                        
                        <div class="default-duration">
                            <div class="duration-info-box">
                                <span class="dashicons dashicons-info-outline"></span>
                                <p>Tüm günler için geçerli olacak standart süreyi aşağıdan seçin. İsterseniz her günün yanındaki dropdown'dan o güne özel farklı bir süre belirleyebilirsiniz.</p>
                            </div>
                            <div class="default-duration-row">
                                <div class="default-duration-label">
                                    <span class="dashicons dashicons-clock"></span> Varsayılan Randevu Süresi
                                </div>
                                <div class="duration-options">
                                    <?php 
                                    $current_duration = isset($settings['slot_duration']) ? $settings['slot_duration'] : 30;
                                    foreach ([15, 30, 45, 60] as $dur): 
                                    ?>
                                    <label class="duration-option <?php echo $current_duration == $dur ? 'active' : ''; ?>">
                                        <input type="radio" name="slot_duration" value="<?php echo $dur; ?>" <?php checked($current_duration, $dur); ?>>
                                        <?php echo $dur; ?> dk
                                    </label>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <button type="submit" class="btn-save">
                    <span class="dashicons dashicons-saved"></span>
                    Kaydet
                </button>
            </form>
        </div>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            // Duration option click
            $('.duration-option').on('click', function() {
                $('.duration-option').removeClass('active');
                $(this).addClass('active');
            });
            
            // Day toggle - add/remove inactive class
            $('.day-toggle input').on('change', function() {
                var $row = $(this).closest('.day-row');
                if ($(this).is(':checked')) {
                    $row.removeClass('inactive');
                } else {
                    $row.addClass('inactive');
                }
            });
        });
        </script>
        <?php
    }

    /**
     * Render Blocked Dates Page
     */
    public function render_blocked_dates_page() {
        if (!current_user_can('manage_options')) {
            wp_die(__('Bu sayfayı görüntüleme yetkiniz yok.', 'appointment-booking'));
        }
        
        $this->db->cleanup_past_blocked_dates();
        $blocked_dates = $this->db->get_blocked_dates_detailed();
        
        $error_messages = array(
            'no_date' => 'Lütfen bir tarih seçin.',
            'past_date' => 'Geçmiş bir tarihi bloklayamazsınız.',
            'already_blocked' => 'Bu tarih zaten tüm gün olarak bloklanmış.',
            'has_appointments' => 'Bu tarihte randevu var! Önce randevuları iptal edin.',
            'no_time' => 'Lütfen başlangıç ve bitiş saatini seçin.',
            'invalid_time_range' => 'Bitiş saati başlangıç saatinden sonra olmalı.',
            'full_day_exists' => 'Bu tarih zaten tüm gün olarak bloklanmış.',
            'overlapping_block' => 'Bu saat aralığı mevcut bir blok ile çakışıyor.',
            'has_appointments_in_range' => 'Bu saat aralığında randevu var! Önce randevuları iptal edin.',
            'db_error' => 'Veritabanı hatası oluştu.'
        );
        ?>
        <style>
        .ab-wrap { margin: 20px 20px 20px 0; font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; }
        .ab-header { display: flex; align-items: center; gap: 12px; margin-bottom: 24px; padding-bottom: 16px; border-bottom: 1px solid #e5e7eb; }
        .ab-header h1 { margin: 0; font-size: 24px; font-weight: 600; color: #111827; display: flex; align-items: center; gap: 10px; }
        .ab-header .dashicons { font-size: 28px; width: 28px; height: 28px; color: #ef4444; }
        .ab-card { background: #fff; border: 1px solid #e5e7eb; border-radius: 12px; margin-bottom: 24px; }
        .ab-card-header { padding: 16px 20px; border-bottom: 1px solid #e5e7eb; display: flex; align-items: center; gap: 10px; }
        .ab-card-header h2 { margin: 0; font-size: 16px; font-weight: 600; color: #111827; }
        .ab-card-body { padding: 20px; }
        .block-form-grid { display: grid; grid-template-columns: 1fr 1fr; gap: 24px; }
        @media (max-width: 768px) { .block-form-grid { grid-template-columns: 1fr; } }
        .form-group { display: flex; flex-direction: column; gap: 10px; }
        .form-group.full-width { grid-column: 1 / -1; }
        .form-group label { font-size: 14px; font-weight: 600; color: #374151; display: flex; align-items: center; gap: 8px; }
        .form-group label .dashicons { font-size: 18px; width: 18px; height: 18px; color: #ef4444; }
        .form-group input[type="text"], .form-group select { padding: 14px 16px; border: 2px solid #e5e7eb; border-radius: 12px; font-size: 14px; background: #f9fafb; transition: all 0.2s; width: 100%; box-sizing: border-box; }
        .form-group input:focus, .form-group select:focus { outline: none; border-color: #ef4444; background: #fff; box-shadow: 0 0 0 4px rgba(239,68,68,0.1); }
        .form-group input::placeholder { color: #9ca3af; }
        .date-picker-wrapper { position: relative; }
        .date-picker-wrapper input { padding: 14px 16px; border: 2px solid #e5e7eb; border-radius: 12px; font-size: 14px; background: #f9fafb; transition: all 0.2s; width: 100%; box-sizing: border-box; cursor: pointer; }
        .date-picker-wrapper input:focus { outline: none; border-color: #ef4444; background: #fff; box-shadow: 0 0 0 4px rgba(239,68,68,0.1); }
        .date-picker-wrapper input.flatpickr-input { display: none; }
        .date-picker-wrapper input.flatpickr-alt-input { display: block !important; }
        .flatpickr-calendar { border-radius: 12px !important; box-shadow: 0 10px 40px rgba(0,0,0,0.15) !important; border: none !important; }
        .flatpickr-day.selected { background: #ef4444 !important; border-color: #ef4444 !important; }
        .flatpickr-day:hover { background: #fee2e2 !important; border-color: #fecaca !important; }
        .flatpickr-day.today { border-color: #ef4444 !important; }
        .block-type-btns { display: flex; gap: 12px; }
        .block-type-btn { flex: 1; padding: 16px 20px; border: 2px solid #e5e7eb; border-radius: 12px; cursor: pointer; font-size: 14px; font-weight: 600; background: #fff; transition: all 0.2s; display: flex; align-items: center; justify-content: center; gap: 8px; }
        .block-type-btn:hover { border-color: #fca5a5; background: #fef2f2; }
        .block-type-btn.active { background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%); color: #fff; border-color: #ef4444; box-shadow: 0 4px 12px rgba(239,68,68,0.3); }
        .block-type-btn input { display: none; }
        .time-inputs { display: none; }
        .time-inputs.show { display: block; margin-top: 20px; padding: 20px; background: #fef2f2; border-radius: 12px; border: 2px solid #fecaca; }
        .time-inputs-label { font-size: 14px; font-weight: 600; color: #991b1b; margin-bottom: 12px; display: flex; align-items: center; gap: 8px; }
        .time-inputs-row { display: flex; gap: 12px; align-items: center; }
        .time-inputs select { padding: 12px 16px; border: 2px solid #fecaca; border-radius: 10px; font-size: 14px; background: #fff; min-width: 120px; }
        .time-inputs select:focus { outline: none; border-color: #ef4444; }
        .time-separator { font-weight: 700; color: #991b1b; font-size: 18px; }
        .reason-section { margin-top: 24px; padding-top: 24px; border-top: 2px solid #f3f4f6; }
        .reason-grid { display: grid; grid-template-columns: repeat(4, 1fr); gap: 10px; margin-bottom: 12px; }
        @media (max-width: 768px) { .reason-grid { grid-template-columns: repeat(2, 1fr); } }
        .reason-btn { padding: 12px 16px; border: 2px solid #e5e7eb; border-radius: 10px; cursor: pointer; font-size: 13px; font-weight: 500; background: #fff; transition: all 0.2s; text-align: center; }
        .reason-btn:hover { border-color: #fca5a5; background: #fef2f2; }
        .reason-btn.active { background: linear-gradient(135deg, #fef2f2 0%, #fee2e2 100%); border-color: #ef4444; color: #991b1b; }
        .custom-reason { margin-top: 12px; }
        .custom-reason input { padding: 14px 16px; border: 2px solid #e5e7eb; border-radius: 12px; font-size: 14px; background: #f9fafb; width: 100%; box-sizing: border-box; }
        .custom-reason input:focus { outline: none; border-color: #ef4444; background: #fff; box-shadow: 0 0 0 4px rgba(239,68,68,0.1); }
        .btn-block { padding: 16px 32px; background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%); color: #fff; border: none; border-radius: 12px; font-size: 15px; font-weight: 600; cursor: pointer; transition: all 0.2s; display: inline-flex; align-items: center; gap: 10px; box-shadow: 0 4px 14px rgba(239,68,68,0.35); margin-top: 24px; }
        .btn-block:hover { transform: translateY(-2px); box-shadow: 0 6px 20px rgba(239,68,68,0.45); }
        .btn-block .dashicons { font-size: 18px; width: 18px; height: 18px; }
        .blocked-list { max-height: 400px; overflow-y: auto; padding-right: 8px; }
        .blocked-list::-webkit-scrollbar { width: 6px; }
        .blocked-list::-webkit-scrollbar-track { background: #fee2e2; border-radius: 3px; }
        .blocked-list::-webkit-scrollbar-thumb { background: #fca5a5; border-radius: 3px; }
        .blocked-list::-webkit-scrollbar-thumb:hover { background: #f87171; }
        .blocked-item { display: flex; align-items: center; justify-content: space-between; padding: 14px 18px; background: linear-gradient(135deg, #fef2f2 0%, #fee2e2 100%); border: 1px solid #fecaca; border-radius: 10px; margin-bottom: 10px; transition: all 0.2s; }
        .blocked-item:hover { transform: translateX(4px); box-shadow: 0 2px 8px rgba(239,68,68,0.15); }
        .blocked-item:last-child { margin-bottom: 0; }
        .blocked-info { display: flex; align-items: center; gap: 12px; flex-wrap: wrap; }
        .blocked-date { font-weight: 600; color: #991b1b; }
        .blocked-time { font-size: 13px; color: #b91c1c; background: #fee2e2; padding: 4px 8px; border-radius: 4px; }
        .blocked-reason { font-size: 13px; color: #6b7280; }
        .btn-unblock { padding: 6px 12px; background: #fff; border: 1px solid #d1d5db; border-radius: 6px; color: #374151; font-size: 13px; cursor: pointer; transition: all 0.2s; }
        .btn-unblock:hover { background: #fef2f2; border-color: #ef4444; color: #ef4444; }
        .empty-state { text-align: center; padding: 40px; color: #6b7280; }
        .popup-overlay { position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.6); display: flex; align-items: center; justify-content: center; z-index: 100000; }
        .popup-modal { background: #fff; border-radius: 16px; padding: 32px; max-width: 400px; text-align: center; }
        .popup-icon { width: 64px; height: 64px; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin: 0 auto 16px; font-size: 28px; }
        .popup-icon.success { background: linear-gradient(135deg, #10b981 0%, #059669 100%); color: #fff; }
        .popup-icon.error { background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%); color: #fff; }
        .popup-title { font-size: 18px; font-weight: 600; margin: 0 0 8px; color: #111827; }
        .popup-message { font-size: 14px; color: #6b7280; margin: 0 0 20px; }
        .popup-btn { padding: 10px 24px; background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%); color: #fff; border: none; border-radius: 8px; font-size: 14px; font-weight: 500; cursor: pointer; }
        </style>
        
        <div class="wrap">
        
        <?php 
        $show_popup = false;
        $popup_type = '';
        $popup_title = '';
        $popup_message = '';
        
        if (isset($_GET['error']) && isset($error_messages[$_GET['error']])) {
            $show_popup = true;
            $popup_type = 'error';
            $popup_title = 'İşlem Başarısız';
            $popup_message = $error_messages[$_GET['error']];
        }
        if (isset($_GET['blocked'])) {
            $show_popup = true;
            $popup_type = 'success';
            $popup_title = 'Blok Eklendi';
            $popup_message = isset($_GET['date']) ? date('d.m.Y', strtotime($_GET['date'])) . ' tarihi bloklandı.' : 'Tarih bloklandı.';
            if (isset($_GET['time_range'])) $popup_message .= ' (' . urldecode($_GET['time_range']) . ')';
        }
        if (isset($_GET['unblocked'])) {
            $show_popup = true;
            $popup_type = 'success';
            $popup_title = 'Blok Kaldırıldı';
            $popup_message = 'Tarih bloğu kaldırıldı.';
        }
        
        if ($show_popup): ?>
        <div class="popup-overlay" id="abPopup">
            <div class="popup-modal">
                <div class="popup-icon <?php echo esc_attr($popup_type); ?>"><?php echo $popup_type === 'success' ? '✓' : '✕'; ?></div>
                <h3 class="popup-title"><?php echo esc_html($popup_title); ?></h3>
                <p class="popup-message"><?php echo esc_html($popup_message); ?></p>
                <button class="popup-btn" onclick="document.getElementById('abPopup').style.display='none'; history.replaceState({}, '', location.pathname + '?page=appointment-booking-blocked');">Tamam</button>
            </div>
        </div>
        <?php endif; ?>
        
        <div class="ab-wrap">
            <div class="ab-header">
                <h1><span class="dashicons dashicons-dismiss"></span> Blokeli Tarihler</h1>
            </div>
            
            <div class="ab-card">
                <div class="ab-card-header">
                    <span class="dashicons dashicons-plus-alt2" style="color: #ef4444;"></span>
                    <h2>Yeni Blok Ekle</h2>
                </div>
                <div class="ab-card-body">
                    <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                        <input type="hidden" name="action" value="appointment_block_date">
                        <?php wp_nonce_field('appointment_settings_nonce'); ?>
                        
                        <div class="block-form-grid">
                            <div class="form-group">
                                <label><span class="dashicons dashicons-calendar-alt"></span> Bloklanacak Tarih</label>
                                <div class="date-picker-wrapper">
                                    <input type="text" name="block_date" id="block_date_picker" required placeholder="Tarih seçin...">
                                </div>
                            </div>
                            
                            <div class="form-group">
                                <label><span class="dashicons dashicons-clock"></span> Blok Tipi</label>
                                <div class="block-type-btns">
                                    <label class="block-type-btn active">
                                        <input type="radio" name="block_type" value="full" checked> 🗓️ Tüm Gün
                                    </label>
                                    <label class="block-type-btn">
                                        <input type="radio" name="block_type" value="partial"> ⏰ Saat Aralığı
                                    </label>
                                </div>
                            </div>
                        </div>
                        
                        <div class="time-inputs" id="timeInputs">
                            <div class="time-inputs-label">
                                <span class="dashicons dashicons-clock"></span> Bloklanacak Saat Aralığı
                            </div>
                            <div class="time-inputs-row">
                                <select name="block_start_time">
                                    <?php for ($h = 8; $h < 22; $h++): ?>
                                    <option value="<?php echo sprintf('%02d:00', $h); ?>"><?php echo sprintf('%02d:00', $h); ?></option>
                                    <?php endfor; ?>
                                </select>
                                <span class="time-separator">→</span>
                                <select name="block_end_time">
                                    <?php for ($h = 8; $h < 22; $h++): ?>
                                    <option value="<?php echo sprintf('%02d:00', $h); ?>" <?php selected($h, 18); ?>><?php echo sprintf('%02d:00', $h); ?></option>
                                    <?php endfor; ?>
                                </select>
                            </div>
                        </div>
                        
                        <div class="reason-section">
                            <label style="font-size: 14px; font-weight: 600; color: #374151; display: flex; align-items: center; gap: 8px; margin-bottom: 14px;">
                                <span class="dashicons dashicons-tag" style="color: #ef4444;"></span> Blok Sebebi (opsiyonel)
                            </label>
                            <div class="reason-grid">
                                <label class="reason-btn" data-reason="Yıllık İzin">
                                    🏖️ Yıllık İzin
                                </label>
                                <label class="reason-btn" data-reason="Resmi Tatil">
                                    🎉 Resmi Tatil
                                </label>
                                <label class="reason-btn" data-reason="Toplantı">
                                    👥 Toplantı
                                </label>
                                <label class="reason-btn" data-reason="Kişisel">
                                    🔒 Kişisel
                                </label>
                            </div>
                            <div class="custom-reason">
                                <input type="text" name="block_reason" id="block_reason" placeholder="Veya özel bir sebep yazın...">
                            </div>
                        </div>
                        
                        <button type="submit" class="btn-block">
                            <span class="dashicons dashicons-lock"></span> Tarihi Blokla
                        </button>
                    </form>
                </div>
            </div>
            
            <div class="ab-card">
                <div class="ab-card-header">
                    <span class="dashicons dashicons-list-view" style="color: #ef4444;"></span>
                    <h2>Mevcut Bloklar</h2>
                </div>
                <div class="ab-card-body">
                    <?php if (empty($blocked_dates)): ?>
                    <div class="empty-state">
                        <span class="dashicons dashicons-calendar-alt" style="font-size: 48px; width: 48px; height: 48px; color: #d1d5db;"></span>
                        <p>Henüz bloklanmış tarih yok.</p>
                    </div>
                    <?php else: ?>
                    <div class="blocked-list">
                        <?php 
                        $months_tr = array('', 'Ocak', 'Şubat', 'Mart', 'Nisan', 'Mayıs', 'Haziran', 'Temmuz', 'Ağustos', 'Eylül', 'Ekim', 'Kasım', 'Aralık');
                        foreach ($blocked_dates as $blocked): 
                            $date_formatted = date('d', strtotime($blocked->blocked_date)) . ' ' . $months_tr[date('n', strtotime($blocked->blocked_date))] . ' ' . date('Y', strtotime($blocked->blocked_date));
                        ?>
                        <div class="blocked-item">
                            <div class="blocked-info">
                                <span class="blocked-date"><?php echo esc_html($date_formatted); ?></span>
                                <?php if (isset($blocked->start_time) && $blocked->start_time): ?>
                                <span class="blocked-time"><?php echo esc_html(substr($blocked->start_time, 0, 5) . ' - ' . substr($blocked->end_time, 0, 5)); ?></span>
                                <?php else: ?>
                                <span class="blocked-time">Tüm Gün</span>
                                <?php endif; ?>
                                <?php if (isset($blocked->reason) && $blocked->reason): ?>
                                <span class="blocked-reason"><?php echo esc_html($blocked->reason); ?></span>
                                <?php endif; ?>
                            </div>
                            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="margin: 0;">
                                <input type="hidden" name="action" value="appointment_unblock_date">
                                <input type="hidden" name="block_id" value="<?php echo esc_attr($blocked->id); ?>">
                                <?php wp_nonce_field('appointment_settings_nonce'); ?>
                                <button type="submit" class="btn-unblock">Kaldır</button>
                            </form>
                        </div>
                        <?php endforeach; ?>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            // Flatpickr date picker
            flatpickr('#block_date_picker', {
                locale: 'tr',
                dateFormat: 'Y-m-d',
                altInput: true,
                altFormat: 'd F Y',
                minDate: 'today',
                disableMobile: true,
                theme: 'light'
            });
            
            // Block type toggle
            $('.block-type-btn').on('click', function() {
                $('.block-type-btn').removeClass('active');
                $(this).addClass('active');
                if ($(this).find('input').val() === 'partial') {
                    $('#timeInputs').addClass('show');
                } else {
                    $('#timeInputs').removeClass('show');
                }
            });
            
            // Reason buttons
            $('.reason-btn').on('click', function() {
                var isActive = $(this).hasClass('active');
                $('.reason-btn').removeClass('active');
                
                if (!isActive) {
                    $(this).addClass('active');
                    $('#block_reason').val($(this).data('reason'));
                } else {
                    $('#block_reason').val('');
                }
            });
            
            // Clear reason button selection when typing custom
            $('#block_reason').on('input', function() {
                if ($(this).val() !== '') {
                    var matchFound = false;
                    $('.reason-btn').each(function() {
                        if ($(this).data('reason') === $('#block_reason').val()) {
                            matchFound = true;
                            $(this).addClass('active');
                        } else {
                            $(this).removeClass('active');
                        }
                    });
                    if (!matchFound) {
                        $('.reason-btn').removeClass('active');
                    }
                }
            });
            
            // ESC key to close popup
            $(document).on('keydown', function(e) {
                if (e.key === 'Escape' && $('#abPopup').length) {
                    $('#abPopup').hide();
                    history.replaceState({}, '', location.pathname + '?page=appointment-booking-blocked');
                }
            });
            
            // Click overlay to close
            $('#abPopup').on('click', function(e) {
                if ($(e.target).hasClass('popup-overlay')) {
                    $(this).hide();
                    history.replaceState({}, '', location.pathname + '?page=appointment-booking-blocked');
                }
            });
        });
        </script>
        <?php
    }

    /**
     * Handle appointment action
     */
    public function handle_appointment_action() {
        check_admin_referer('appointment_action_nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die(__('Yetkiniz yok.', 'appointment-booking'));
        }
        
        $action = isset($_POST['appointment_action']) ? sanitize_text_field($_POST['appointment_action']) : '';
        $id = isset($_POST['appointment_id']) ? absint($_POST['appointment_id']) : 0;
        
        if ($action && $id) {
            switch ($action) {
                case 'confirm':
                    $this->db->update_appointment_status($id, 'confirmed');
                    break;
                case 'reject':
                    $this->db->update_appointment_status($id, 'rejected');
                    break;
                case 'delete':
                    $this->db->delete_appointment($id);
                    break;
            }
        }
        
        wp_redirect(admin_url('admin.php?page=appointment-booking'));
        exit;
    }
    
    /**
     * Handle working hours save
     */
    public function handle_save_working_hours() {
        check_admin_referer('appointment_settings_nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die(__('Yetkiniz yok.', 'appointment-booking'));
        }
        
        $settings = get_option('appointment_booking_settings', array());
        $default_duration = isset($_POST['slot_duration']) ? absint($_POST['slot_duration']) : 30;
        $settings['slot_duration'] = $default_duration;
        update_option('appointment_booking_settings', $settings);
        
        $days = isset($_POST['days']) ? $_POST['days'] : array();
        
        global $wpdb;
        $slots_table = $wpdb->prefix . 'appointment_slots';
        $wpdb->query("DELETE FROM {$slots_table}");
        
        foreach ($days as $day_num => $day_data) {
            if (empty($day_data['active'])) {
                continue;
            }
            
            if (isset($day_data['ranges']) && is_array($day_data['ranges'])) {
                foreach ($day_data['ranges'] as $range) {
                    if (empty($range['start']) || empty($range['end'])) {
                        continue;
                    }
                    
                    $wpdb->insert(
                        $slots_table,
                        array(
                            'day_of_week' => absint($day_num),
                            'start_time' => sanitize_text_field($range['start']) . ':00',
                            'end_time' => sanitize_text_field($range['end']) . ':00',
                            'slot_duration' => isset($range['duration']) ? absint($range['duration']) : $default_duration,
                            'is_active' => 1
                        ),
                        array('%d', '%s', '%s', '%d', '%d')
                    );
                }
            }
        }
        
        wp_redirect(admin_url('admin.php?page=appointment-booking-hours&updated=true'));
        exit;
    }
    
    /**
     * Handle block date
     */
    public function handle_block_date() {
        check_admin_referer('appointment_settings_nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die(__('Yetkiniz yok.', 'appointment-booking'));
        }
        
        $date = isset($_POST['block_date']) ? sanitize_text_field($_POST['block_date']) : '';
        $block_type = isset($_POST['block_type']) ? sanitize_text_field($_POST['block_type']) : 'full';
        
        if (empty($date)) {
            wp_redirect(admin_url('admin.php?page=appointment-booking-blocked&error=no_date'));
            exit;
        }
        
        if (strtotime($date) < strtotime(date('Y-m-d'))) {
            wp_redirect(admin_url('admin.php?page=appointment-booking-blocked&error=past_date'));
            exit;
        }
        
        $start_time = null;
        $end_time = null;
        $reason = isset($_POST['block_reason']) ? sanitize_text_field($_POST['block_reason']) : '';
        
        if ($block_type === 'full') {
            if ($this->db->has_full_day_block($date)) {
                wp_redirect(admin_url('admin.php?page=appointment-booking-blocked&error=already_blocked'));
                exit;
            }
            
            $appointment_count = $this->db->get_appointments_count_on_date($date);
            if ($appointment_count > 0) {
                wp_redirect(admin_url('admin.php?page=appointment-booking-blocked&error=has_appointments&count=' . $appointment_count));
                exit;
            }
        } else {
            $start_time = isset($_POST['block_start_time']) && !empty($_POST['block_start_time']) ? sanitize_text_field($_POST['block_start_time']) . ':00' : null;
            $end_time = isset($_POST['block_end_time']) && !empty($_POST['block_end_time']) ? sanitize_text_field($_POST['block_end_time']) . ':00' : null;
            
            if (empty($start_time) || empty($end_time)) {
                wp_redirect(admin_url('admin.php?page=appointment-booking-blocked&error=no_time'));
                exit;
            }
            
            if (strtotime($end_time) <= strtotime($start_time)) {
                wp_redirect(admin_url('admin.php?page=appointment-booking-blocked&error=invalid_time_range'));
                exit;
            }
            
            if ($this->db->has_full_day_block($date)) {
                wp_redirect(admin_url('admin.php?page=appointment-booking-blocked&error=full_day_exists'));
                exit;
            }
            
            if ($this->db->has_overlapping_block($date, $start_time, $end_time)) {
                wp_redirect(admin_url('admin.php?page=appointment-booking-blocked&error=overlapping_block'));
                exit;
            }
            
            if ($this->db->has_appointments_in_range($date, $start_time, $end_time)) {
                wp_redirect(admin_url('admin.php?page=appointment-booking-blocked&error=has_appointments_in_range'));
                exit;
            }
        }
        
        $result = $this->db->block_date($date, $reason, $start_time, $end_time);
        
        if ($result) {
            $redirect_url = admin_url('admin.php?page=appointment-booking-blocked&blocked=true&date=' . urlencode($date));
            if ($start_time && $end_time) {
                $time_range = substr($start_time, 0, 5) . ' - ' . substr($end_time, 0, 5);
                $redirect_url .= '&time_range=' . urlencode($time_range);
            } else {
                $redirect_url .= '&time_range=' . urlencode('Tüm Gün');
            }
            wp_redirect($redirect_url);
        } else {
            wp_redirect(admin_url('admin.php?page=appointment-booking-blocked&error=db_error'));
        }
        exit;
    }
    
    /**
     * Handle unblock date
     */
    public function handle_unblock_date() {
        check_admin_referer('appointment_settings_nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die(__('Yetkiniz yok.', 'appointment-booking'));
        }
        
        $block_id = isset($_POST['block_id']) ? absint($_POST['block_id']) : 0;
        
        if ($block_id) {
            $this->db->delete_blocked_date($block_id);
        }
        
        wp_redirect(admin_url('admin.php?page=appointment-booking-blocked&unblocked=true'));
        exit;
    }
    
    /**
     * AJAX: Update appointment status
     */
    public function ajax_update_status() {
        check_ajax_referer('appointment_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Yetkiniz yok.'));
        }
        
        $id = isset($_POST['id']) ? absint($_POST['id']) : 0;
        $status = isset($_POST['status']) ? sanitize_text_field($_POST['status']) : '';
        
        if ($id && $status) {
            $result = $this->db->update_appointment_status($id, $status);
            if ($result) {
                wp_send_json_success(array('message' => 'Durum güncellendi.'));
            }
        }
        
        wp_send_json_error(array('message' => 'Güncelleme başarısız.'));
    }
    
    /**
     * AJAX: Delete appointment
     */
    public function ajax_delete_appointment() {
        check_ajax_referer('appointment_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Yetkiniz yok.'));
        }
        
        $id = isset($_POST['id']) ? absint($_POST['id']) : 0;
        
        if ($id) {
            $result = $this->db->delete_appointment($id);
            if ($result) {
                wp_send_json_success(array('message' => 'Randevu silindi.'));
            }
        }
        
        wp_send_json_error(array('message' => 'Silme başarısız.'));
    }
    
    /**
     * Handle email settings save
     */
    public function handle_save_email_settings() {
        check_admin_referer('appointment_email_settings_nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die(__('Yetkiniz yok.', 'appointment-booking'));
        }
        
        $email_settings = array(
            'enabled' => isset($_POST['notifications_enabled']) ? true : false,
            'admin_email' => isset($_POST['admin_email']) ? sanitize_email($_POST['admin_email']) : '',
            'from_name' => isset($_POST['from_name']) ? sanitize_text_field($_POST['from_name']) : '',
            'from_email' => isset($_POST['from_email']) ? sanitize_email($_POST['from_email']) : '',
            'smtp_enabled' => isset($_POST['smtp_enabled']) ? true : false,
            'smtp_host' => isset($_POST['smtp_host']) ? sanitize_text_field($_POST['smtp_host']) : '',
            'smtp_port' => isset($_POST['smtp_port']) ? absint($_POST['smtp_port']) : 587,
            'smtp_user' => isset($_POST['smtp_user']) ? sanitize_text_field($_POST['smtp_user']) : '',
            'smtp_pass' => isset($_POST['smtp_pass']) ? $_POST['smtp_pass'] : '',
            'smtp_secure' => isset($_POST['smtp_secure']) ? sanitize_text_field($_POST['smtp_secure']) : 'tls'
        );
        
        update_option('appointment_booking_email_settings', $email_settings);
        
        wp_redirect(admin_url('admin.php?page=appointment-booking-settings&email_saved=true'));
        exit;
    }
    
    /**
     * Handle custom settings save (appearance)
     */
    public function handle_save_custom_settings() {
        check_admin_referer('appointment_custom_settings_nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die(__('Yetkiniz yok.', 'appointment-booking'));
        }
        
        $custom_settings = array(
            'primary_color' => isset($_POST['primary_color']) ? sanitize_hex_color($_POST['primary_color']) : '#2563eb',
            'font_family' => isset($_POST['font_family']) ? stripslashes(wp_strip_all_tags($_POST['font_family'])) : '',
            'font_size' => isset($_POST['font_size']) ? sanitize_text_field($_POST['font_size']) : 'normal'
        );
        
        update_option('appointment_booking_custom', $custom_settings);
        
        wp_redirect(admin_url('admin.php?page=appointment-booking-settings&custom_saved=true'));
        exit;
    }
    
    /**
     * AJAX: Test email
     */
    public function ajax_test_email() {
        check_ajax_referer('appointment_test_email_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Yetkiniz yok.'));
        }
        
        $email_settings = get_option('appointment_booking_email_settings', array());
        $to = isset($email_settings['admin_email']) && $email_settings['admin_email'] ? $email_settings['admin_email'] : get_option('admin_email');
        $from_name = isset($email_settings['from_name']) && $email_settings['from_name'] ? $email_settings['from_name'] : get_bloginfo('name');
        $from_email = isset($email_settings['from_email']) && $email_settings['from_email'] ? $email_settings['from_email'] : get_option('admin_email');
        
        $subject = 'Randevu Eklentisi - Test E-postası';
        $message = "Bu bir test e-postasıdır.\n\nE-posta ayarlarınız doğru çalışıyor.\n\nRandevu Eklentisi";
        
        $headers = array(
            'Content-Type: text/plain; charset=UTF-8',
            'From: ' . $from_name . ' <' . $from_email . '>'
        );
        
        // If SMTP enabled, configure PHPMailer
        if (isset($email_settings['smtp_enabled']) && $email_settings['smtp_enabled']) {
            add_action('phpmailer_init', function($phpmailer) use ($email_settings) {
                $phpmailer->isSMTP();
                $phpmailer->Host = $email_settings['smtp_host'];
                $phpmailer->Port = $email_settings['smtp_port'];
                $phpmailer->SMTPAuth = true;
                $phpmailer->Username = $email_settings['smtp_user'];
                $phpmailer->Password = $email_settings['smtp_pass'];
                $phpmailer->SMTPSecure = $email_settings['smtp_secure'];
            });
        }
        
        $sent = wp_mail($to, $subject, $message, $headers);
        
        if ($sent) {
            wp_send_json_success(array('message' => 'Test e-postası gönderildi.'));
        } else {
            wp_send_json_error(array('message' => 'E-posta gönderilemedi. Ayarları kontrol edin.'));
        }
    }
    
    /**
     * Render Calendar Page - Monthly view with appointments and blocked dates
     */
    public function render_calendar_page() {
        if (!current_user_can('manage_options')) {
            wp_die(__('Bu sayfayı görüntüleme yetkiniz yok.', 'appointment-booking'));
        }
        
        // Get current month/year from URL or use current
        $year = isset($_GET['year']) ? absint($_GET['year']) : date('Y');
        $month = isset($_GET['month']) ? absint($_GET['month']) : date('n');
        
        // Validate
        if ($month < 1) { $month = 12; $year--; }
        if ($month > 12) { $month = 1; $year++; }
        
        // Get data for this month
        $first_day = sprintf('%04d-%02d-01', $year, $month);
        $last_day = date('Y-m-t', strtotime($first_day));
        
        // Get appointments for this month
        global $wpdb;
        $appointments_table = $wpdb->prefix . 'appointments';
        $blocked_table = $wpdb->prefix . 'appointment_blocked_dates';
        
        $appointments = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$appointments_table} 
             WHERE appointment_date BETWEEN %s AND %s 
             ORDER BY appointment_date, appointment_time",
            $first_day, $last_day
        ));
        
        // Get blocked dates for this month
        $blocked_dates = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$blocked_table} 
             WHERE blocked_date BETWEEN %s AND %s 
             ORDER BY blocked_date, start_time",
            $first_day, $last_day
        ));
        
        // Organize data by date
        $calendar_data = array();
        foreach ($appointments as $apt) {
            $date = $apt->appointment_date;
            if (!isset($calendar_data[$date])) {
                $calendar_data[$date] = array('appointments' => array(), 'blocked' => array());
            }
            $calendar_data[$date]['appointments'][] = $apt;
        }
        foreach ($blocked_dates as $block) {
            $date = $block->blocked_date;
            if (!isset($calendar_data[$date])) {
                $calendar_data[$date] = array('appointments' => array(), 'blocked' => array());
            }
            $calendar_data[$date]['blocked'][] = $block;
        }
        
        // Month names in Turkish
        $months_tr = array('', 'Ocak', 'Şubat', 'Mart', 'Nisan', 'Mayıs', 'Haziran', 'Temmuz', 'Ağustos', 'Eylül', 'Ekim', 'Kasım', 'Aralık');
        $days_tr = array('Pzt', 'Sal', 'Çar', 'Per', 'Cum', 'Cmt', 'Paz');
        
        // Calculate calendar grid
        $days_in_month = date('t', strtotime($first_day));
        $first_day_weekday = date('N', strtotime($first_day)); // 1=Monday, 7=Sunday
        
        // Navigation URLs
        $prev_month = $month - 1;
        $prev_year = $year;
        if ($prev_month < 1) { $prev_month = 12; $prev_year--; }
        
        $next_month = $month + 1;
        $next_year = $year;
        if ($next_month > 12) { $next_month = 1; $next_year++; }
        
        $prev_url = admin_url("admin.php?page=appointment-booking-calendar&year={$prev_year}&month={$prev_month}");
        $next_url = admin_url("admin.php?page=appointment-booking-calendar&year={$next_year}&month={$next_month}");
        $today_url = admin_url("admin.php?page=appointment-booking-calendar");
        ?>
        <style>
        .cal-wrap { margin: 20px 20px 20px 0; font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; }
        .cal-header { display: flex; align-items: center; justify-content: space-between; margin-bottom: 24px; padding-bottom: 16px; border-bottom: 1px solid #e5e7eb; }
        .cal-header h1 { margin: 0; font-size: 24px; font-weight: 600; color: #111827; display: flex; align-items: center; gap: 10px; }
        .cal-header .dashicons { font-size: 28px; width: 28px; height: 28px; color: #8b5cf6; }
        .cal-nav { display: flex; align-items: center; gap: 12px; }
        .cal-nav-btn { display: inline-flex; align-items: center; justify-content: center; width: 40px; height: 40px; background: #fff; border: 2px solid #e5e7eb; border-radius: 10px; color: #374151; font-size: 20px; text-decoration: none; transition: all 0.2s; }
        .cal-nav-btn:hover { border-color: #8b5cf6; color: #8b5cf6; background: #f5f3ff; }
        .cal-month-year { font-size: 20px; font-weight: 700; color: #111827; min-width: 180px; text-align: center; }
        .cal-today-btn { padding: 10px 20px; background: linear-gradient(135deg, #8b5cf6 0%, #7c3aed 100%); color: #fff; border: none; border-radius: 10px; font-size: 14px; font-weight: 600; text-decoration: none; transition: all 0.2s; }
        .cal-today-btn:hover { color: #fff; transform: translateY(-2px); box-shadow: 0 4px 12px rgba(139,92,246,0.4); }
        
        .cal-legend { display: flex; gap: 20px; margin-bottom: 20px; flex-wrap: wrap; }
        .cal-legend-item { display: flex; align-items: center; gap: 8px; font-size: 13px; color: #6b7280; }
        .cal-legend-dot { width: 12px; height: 12px; border-radius: 4px; }
        .cal-legend-dot.pending { background: #fbbf24; }
        .cal-legend-dot.confirmed { background: #10b981; }
        .cal-legend-dot.rejected { background: #ef4444; }
        .cal-legend-dot.blocked { background: #1f2937; }
        
        .cal-grid { display: grid; grid-template-columns: repeat(7, 1fr); gap: 1px; background: #e5e7eb; border: 1px solid #e5e7eb; border-radius: 16px; overflow: hidden; }
        .cal-weekday { padding: 14px 8px; text-align: center; font-size: 12px; font-weight: 700; color: #6b7280; background: #f9fafb; text-transform: uppercase; letter-spacing: 0.5px; }
        .cal-day { min-height: 120px; padding: 8px; background: #fff; vertical-align: top; position: relative; }
        .cal-day.empty { background: #f9fafb; }
        .cal-day.today { background: #f5f3ff; }
        .cal-day.today .cal-day-num { background: #8b5cf6; color: #fff; }
        .cal-day-num { display: inline-flex; align-items: center; justify-content: center; width: 28px; height: 28px; font-size: 13px; font-weight: 600; color: #374151; border-radius: 8px; margin-bottom: 6px; }
        .cal-day.weekend .cal-day-num { color: #9ca3af; }
        
        .cal-events { display: flex; flex-direction: column; gap: 4px; }
        .cal-event { padding: 4px 8px; border-radius: 6px; font-size: 11px; font-weight: 500; cursor: pointer; transition: all 0.15s; white-space: nowrap; overflow: hidden; text-overflow: ellipsis; }
        .cal-event:hover { transform: scale(1.02); }
        .cal-event.pending { background: #fef3c7; color: #92400e; border-left: 3px solid #fbbf24; }
        .cal-event.confirmed { background: #d1fae5; color: #065f46; border-left: 3px solid #10b981; }
        .cal-event.rejected { background: #fee2e2; color: #991b1b; border-left: 3px solid #ef4444; }
        .cal-event.blocked { background: #1f2937; color: #fff; border-left: 3px solid #000; }
        .cal-event-time { font-weight: 700; }
        .cal-event-more { font-size: 11px; color: #6b7280; padding: 2px 8px; cursor: pointer; }
        .cal-event-more:hover { color: #8b5cf6; }
        
        /* Day detail popup */
        .cal-popup-overlay { display: none; position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.5); z-index: 9999; align-items: center; justify-content: center; }
        .cal-popup-overlay.show { display: flex; }
        .cal-popup { background: #fff; border-radius: 16px; width: 90%; max-width: 480px; max-height: 80vh; overflow: hidden; box-shadow: 0 25px 50px rgba(0,0,0,0.25); }
        .cal-popup-header { padding: 20px 24px; background: linear-gradient(135deg, #8b5cf6 0%, #7c3aed 100%); color: #fff; display: flex; align-items: center; justify-content: space-between; }
        .cal-popup-header h3 { margin: 0; font-size: 18px; font-weight: 600; }
        .cal-popup-close { background: rgba(255,255,255,0.2); border: none; color: #fff; width: 32px; height: 32px; border-radius: 8px; font-size: 20px; cursor: pointer; transition: all 0.2s; }
        .cal-popup-close:hover { background: rgba(255,255,255,0.3); }
        .cal-popup-body { padding: 20px 24px; max-height: 60vh; overflow-y: auto; }
        .cal-popup-section { margin-bottom: 20px; }
        .cal-popup-section:last-child { margin-bottom: 0; }
        .cal-popup-section h4 { margin: 0 0 12px; font-size: 13px; font-weight: 700; color: #6b7280; text-transform: uppercase; letter-spacing: 0.5px; display: flex; align-items: center; gap: 8px; }
        .cal-popup-item { display: flex; align-items: center; gap: 12px; padding: 12px; background: #f9fafb; border-radius: 10px; margin-bottom: 8px; }
        .cal-popup-item:last-child { margin-bottom: 0; }
        .cal-popup-time { font-size: 14px; font-weight: 700; color: #111827; min-width: 50px; }
        .cal-popup-info { flex: 1; }
        .cal-popup-name { font-size: 14px; font-weight: 600; color: #111827; }
        .cal-popup-phone { font-size: 13px; color: #6b7280; display: flex; align-items: center; gap: 4px; margin-top: 2px; }
        .cal-popup-status { padding: 4px 10px; border-radius: 6px; font-size: 11px; font-weight: 600; }
        .cal-popup-status.pending { background: #fef3c7; color: #92400e; }
        .cal-popup-status.confirmed { background: #d1fae5; color: #065f46; }
        .cal-popup-status.rejected { background: #fee2e2; color: #991b1b; }
        .cal-popup-blocked { padding: 12px; background: #1f2937; color: #fff; border-radius: 10px; margin-bottom: 8px; }
        .cal-popup-blocked-time { font-weight: 700; }
        .cal-popup-blocked-reason { font-size: 12px; opacity: 0.8; margin-top: 4px; }
        .cal-popup-empty { text-align: center; padding: 20px; color: #9ca3af; font-size: 14px; }
        
        @media (max-width: 900px) {
            .cal-day { min-height: 80px; padding: 4px; }
            .cal-day-num { width: 24px; height: 24px; font-size: 11px; }
            .cal-event { font-size: 10px; padding: 2px 4px; }
            .cal-weekday { padding: 10px 4px; font-size: 10px; }
        }
        </style>
        
        <div class="wrap">
        <div class="cal-wrap">
            <div class="cal-header">
                <h1><span class="dashicons dashicons-calendar-alt"></span> Takvimim</h1>
                <div class="cal-nav">
                    <a href="<?php echo esc_url($prev_url); ?>" class="cal-nav-btn">‹</a>
                    <span class="cal-month-year"><?php echo $months_tr[$month] . ' ' . $year; ?></span>
                    <a href="<?php echo esc_url($next_url); ?>" class="cal-nav-btn">›</a>
                    <a href="<?php echo esc_url($today_url); ?>" class="cal-today-btn">Bugün</a>
                </div>
            </div>
            
            <div class="cal-legend">
                <div class="cal-legend-item"><span class="cal-legend-dot pending"></span> Bekleyen</div>
                <div class="cal-legend-item"><span class="cal-legend-dot confirmed"></span> Onaylı</div>
                <div class="cal-legend-item"><span class="cal-legend-dot rejected"></span> Reddedildi</div>
                <div class="cal-legend-item"><span class="cal-legend-dot blocked"></span> Bloklu</div>
            </div>
            
            <div class="cal-grid">
                <?php foreach ($days_tr as $day): ?>
                <div class="cal-weekday"><?php echo $day; ?></div>
                <?php endforeach; ?>
                
                <?php 
                // Empty cells before first day
                for ($i = 1; $i < $first_day_weekday; $i++): ?>
                <div class="cal-day empty"></div>
                <?php endfor; ?>
                
                <?php 
                // Days of month
                $today = date('Y-m-d');
                for ($day = 1; $day <= $days_in_month; $day++): 
                    $date = sprintf('%04d-%02d-%02d', $year, $month, $day);
                    $day_of_week = date('N', strtotime($date));
                    $is_weekend = ($day_of_week >= 6);
                    $is_today = ($date === $today);
                    $day_data = isset($calendar_data[$date]) ? $calendar_data[$date] : array('appointments' => array(), 'blocked' => array());
                    
                    $classes = 'cal-day';
                    if ($is_weekend) $classes .= ' weekend';
                    if ($is_today) $classes .= ' today';
                ?>
                <div class="<?php echo $classes; ?>" data-date="<?php echo $date; ?>">
                    <span class="cal-day-num"><?php echo $day; ?></span>
                    <div class="cal-events">
                        <?php 
                        $shown = 0;
                        $total = count($day_data['appointments']) + count($day_data['blocked']);
                        
                        // Show blocked dates first
                        foreach ($day_data['blocked'] as $block): 
                            if ($shown >= 3) break;
                            $shown++;
                            $time_text = $block->start_time ? substr($block->start_time, 0, 5) : 'Tüm gün';
                        ?>
                        <div class="cal-event blocked" title="<?php echo esc_attr($block->reason ?: 'Bloklu'); ?>">
                            <span class="cal-event-time"><?php echo $time_text; ?></span> 🚫
                        </div>
                        <?php endforeach; ?>
                        
                        <?php 
                        // Show appointments
                        foreach ($day_data['appointments'] as $apt): 
                            if ($shown >= 3) break;
                            $shown++;
                        ?>
                        <div class="cal-event <?php echo esc_attr($apt->status); ?>" title="<?php echo esc_attr($apt->user_name . ' ' . $apt->user_surname); ?>">
                            <span class="cal-event-time"><?php echo substr($apt->appointment_time, 0, 5); ?></span>
                            <?php echo esc_html(mb_substr($apt->user_name, 0, 1) . '. ' . $apt->user_surname); ?>
                        </div>
                        <?php endforeach; ?>
                        
                        <?php if ($total > 3): ?>
                        <div class="cal-event-more" onclick="showDayDetail('<?php echo $date; ?>')">+<?php echo ($total - 3); ?> daha</div>
                        <?php endif; ?>
                    </div>
                </div>
                <?php endfor; ?>
                
                <?php 
                // Empty cells after last day
                $last_day_weekday = date('N', strtotime($last_day));
                for ($i = $last_day_weekday; $i < 7; $i++): ?>
                <div class="cal-day empty"></div>
                <?php endfor; ?>
            </div>
        </div>
        </div>
        
        <!-- Day Detail Popup -->
        <div class="cal-popup-overlay" id="calPopup">
            <div class="cal-popup">
                <div class="cal-popup-header">
                    <h3 id="popupTitle">Gün Detayı</h3>
                    <button class="cal-popup-close" onclick="closePopup()">×</button>
                </div>
                <div class="cal-popup-body" id="popupBody">
                    <!-- Content loaded dynamically -->
                </div>
            </div>
        </div>
        
        <script>
        // Store calendar data for popup
        var calendarData = <?php echo json_encode($calendar_data); ?>;
        var monthsTr = ['', 'Ocak', 'Şubat', 'Mart', 'Nisan', 'Mayıs', 'Haziran', 'Temmuz', 'Ağustos', 'Eylül', 'Ekim', 'Kasım', 'Aralık'];
        
        // Format phone number: (532) 000 00 00
        function formatPhone(phone) {
            if (!phone) return '';
            var digits = phone.replace(/\D/g, '');
            // Remove leading 0 or 90
            if (digits.startsWith('90')) digits = digits.slice(2);
            if (digits.startsWith('0')) digits = digits.slice(1);
            // Format: (532) 000 00 00
            if (digits.length === 10) {
                return '(' + digits.slice(0,3) + ') ' + digits.slice(3,6) + ' ' + digits.slice(6,8) + ' ' + digits.slice(8,10);
            }
            return phone;
        }
        
        // Click on day to show detail
        document.querySelectorAll('.cal-day:not(.empty)').forEach(function(day) {
            day.style.cursor = 'pointer';
            day.addEventListener('click', function() {
                var date = this.getAttribute('data-date');
                showDayDetail(date);
            });
        });
        
        function showDayDetail(date) {
            var data = calendarData[date] || {appointments: [], blocked: []};
            var dateParts = date.split('-');
            var dateFormatted = parseInt(dateParts[2]) + ' ' + monthsTr[parseInt(dateParts[1])] + ' ' + dateParts[0];
            
            document.getElementById('popupTitle').textContent = dateFormatted;
            
            var html = '';
            
            // Blocked section
            if (data.blocked && data.blocked.length > 0) {
                html += '<div class="cal-popup-section">';
                html += '<h4>🚫 Bloklu Saatler</h4>';
                data.blocked.forEach(function(block) {
                    var timeText = block.start_time ? block.start_time.substr(0,5) + ' - ' + block.end_time.substr(0,5) : 'Tüm Gün';
                    html += '<div class="cal-popup-blocked">';
                    html += '<div class="cal-popup-blocked-time">' + timeText + '</div>';
                    if (block.reason) {
                        html += '<div class="cal-popup-blocked-reason">' + block.reason + '</div>';
                    }
                    html += '</div>';
                });
                html += '</div>';
            }
            
            // Appointments section
            if (data.appointments && data.appointments.length > 0) {
                html += '<div class="cal-popup-section">';
                html += '<h4>📅 Randevular</h4>';
                data.appointments.forEach(function(apt) {
                    var statusText = {pending: 'Bekliyor', confirmed: 'Onaylı', rejected: 'Reddedildi'}[apt.status] || apt.status;
                    html += '<div class="cal-popup-item">';
                    html += '<div class="cal-popup-time">' + apt.appointment_time.substr(0,5) + '</div>';
                    html += '<div class="cal-popup-info">';
                    html += '<div class="cal-popup-name">' + apt.user_name + ' ' + apt.user_surname + '</div>';
                    html += '<div class="cal-popup-phone">📞 ' + formatPhone(apt.user_phone) + '</div>';
                    html += '</div>';
                    html += '<span class="cal-popup-status ' + apt.status + '">' + statusText + '</span>';
                    html += '</div>';
                });
                html += '</div>';
            }
            
            if (!html) {
                html = '<div class="cal-popup-empty">Bu tarihte randevu veya blok yok.</div>';
            }
            
            document.getElementById('popupBody').innerHTML = html;
            document.getElementById('calPopup').classList.add('show');
        }
        
        function closePopup() {
            document.getElementById('calPopup').classList.remove('show');
        }
        
        // Close on overlay click
        document.getElementById('calPopup').addEventListener('click', function(e) {
            if (e.target === this) closePopup();
        });
        
        // Close on ESC
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') closePopup();
        });
        </script>
        <?php
    }
    
    /**
     * Render clients page
     */
    public function render_clients_page() {
        if (!current_user_can('manage_options')) {
            wp_die(__('Bu sayfayı görüntüleme yetkiniz yok.', 'appointment-booking'));
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'appointments';
        
        // Pagination
        $per_page = 20;
        $current_page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;
        $offset = ($current_page - 1) * $per_page;
        
        // Filters
        $search = isset($_GET['s']) ? sanitize_text_field($_GET['s']) : '';
        $status_filter = isset($_GET['status']) ? sanitize_text_field($_GET['status']) : '';
        
        // Build query
        $where = array('1=1');
        if ($search) {
            $where[] = $wpdb->prepare(
                "(user_name LIKE %s OR user_surname LIKE %s OR user_phone LIKE %s)",
                '%' . $wpdb->esc_like($search) . '%',
                '%' . $wpdb->esc_like($search) . '%',
                '%' . $wpdb->esc_like($search) . '%'
            );
        }
        if ($status_filter) {
            $where[] = $wpdb->prepare("status = %s", $status_filter);
        }
        
        $where_sql = implode(' AND ', $where);
        
        // Get unique clients with their latest appointment
        $clients_query = "
            SELECT 
                user_phone,
                user_name,
                user_surname,
                MAX(appointment_date) as last_appointment_date,
                MAX(appointment_time) as last_appointment_time,
                MAX(id) as latest_id,
                COUNT(*) as total_appointments,
                SUM(CASE WHEN status = 'confirmed' THEN 1 ELSE 0 END) as confirmed_count,
                SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending_count,
                SUM(CASE WHEN status = 'rejected' THEN 1 ELSE 0 END) as rejected_count
            FROM {$table}
            WHERE {$where_sql}
            GROUP BY user_phone
            ORDER BY latest_id DESC
            LIMIT %d OFFSET %d
        ";
        
        $clients = $wpdb->get_results($wpdb->prepare($clients_query, $per_page, $offset));
        
        // Total count
        $total_query = "SELECT COUNT(DISTINCT user_phone) FROM {$table} WHERE {$where_sql}";
        $total_clients = $wpdb->get_var($total_query);
        $total_pages = ceil($total_clients / $per_page);
        
        ?>
        <style>
        .clients-wrap { margin: 20px 20px 20px 0; font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; }
        .clients-header { display: flex; align-items: center; justify-content: space-between; margin-bottom: 20px; padding-bottom: 16px; border-bottom: 2px solid #e5e7eb; }
        .clients-header h1 { margin: 0; font-size: 24px; font-weight: 600; color: #111827; display: flex; align-items: center; gap: 10px; }
        .clients-header .dashicons { font-size: 28px; width: 28px; height: 28px; color: #3b82f6; }
        .clients-stats { display: grid; grid-template-columns: repeat(4, 1fr); gap: 16px; margin-bottom: 20px; }
        .stat-card { background: linear-gradient(135deg, #fff 0%, #f8fafc 100%); border: 2px solid #e5e7eb; border-radius: 12px; padding: 16px; }
        .stat-card .stat-value { font-size: 28px; font-weight: 700; color: #111827; margin-bottom: 4px; }
        .stat-card .stat-label { font-size: 13px; color: #6b7280; }
        .clients-filters { background: #fff; border: 2px solid #e5e7eb; border-radius: 12px; padding: 16px; margin-bottom: 20px; display: flex; gap: 12px; align-items: center; flex-wrap: wrap; }
        .filter-input { padding: 10px 14px; border: 2px solid #e5e7eb; border-radius: 8px; font-size: 14px; min-width: 250px; }
        .filter-select { padding: 10px 14px; border: 2px solid #e5e7eb; border-radius: 8px; font-size: 14px; }
        .filter-btn { padding: 10px 20px; background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%); color: #fff; border: none; border-radius: 8px; font-size: 14px; font-weight: 600; cursor: pointer; }
        .filter-btn:hover { transform: translateY(-1px); box-shadow: 0 4px 12px rgba(59,130,246,0.3); }
        .clients-table { background: #fff; border: 2px solid #e5e7eb; border-radius: 12px; overflow: hidden; }
        .clients-table table { width: 100%; border-collapse: collapse; }
        .clients-table th { background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%); padding: 14px 16px; text-align: left; font-size: 13px; font-weight: 600; color: #374151; border-bottom: 2px solid #e5e7eb; }
        .clients-table td { padding: 14px 16px; border-bottom: 1px solid #f3f4f6; font-size: 14px; color: #1f2937; }
        .clients-table tr:hover { background: #f9fafb; }
        .client-name { font-weight: 600; color: #111827; }
        .client-phone { color: #6b7280; font-size: 13px; }
        .appointment-badge { display: inline-block; padding: 4px 10px; border-radius: 6px; font-size: 12px; font-weight: 600; }
        .badge-confirmed { background: #d1fae5; color: #065f46; }
        .badge-pending { background: #fef3c7; color: #92400e; }
        .badge-rejected { background: #fee2e2; color: #991b1b; }
        .pagination { display: flex; justify-content: center; gap: 8px; margin-top: 20px; }
        .pagination a, .pagination span { padding: 8px 14px; background: #fff; border: 2px solid #e5e7eb; border-radius: 8px; font-size: 14px; font-weight: 600; color: #374151; text-decoration: none; }
        .pagination a:hover { background: #3b82f6; color: #fff; border-color: #3b82f6; }
        .pagination .current { background: #3b82f6; color: #fff; border-color: #3b82f6; }
        </style>
        
        <div class="wrap">
        <div class="clients-wrap">
            <div class="clients-header">
                <h1><span class="dashicons dashicons-groups"></span> Danışanlar</h1>
            </div>
            
            <div class="clients-stats">
                <div class="stat-card">
                    <div class="stat-value"><?php echo number_format($total_clients); ?></div>
                    <div class="stat-label">Toplam Danışan</div>
                </div>
                <div class="stat-card">
                    <div class="stat-value"><?php echo number_format($wpdb->get_var("SELECT COUNT(*) FROM {$table} WHERE status='confirmed'")); ?></div>
                    <div class="stat-label">Onaylanan Randevular</div>
                </div>
                <div class="stat-card">
                    <div class="stat-value"><?php echo number_format($wpdb->get_var("SELECT COUNT(*) FROM {$table} WHERE status='pending'")); ?></div>
                    <div class="stat-label">Bekleyen Randevular</div>
                </div>
                <div class="stat-card">
                    <div class="stat-value"><?php echo number_format($wpdb->get_var("SELECT COUNT(*) FROM {$table} WHERE status='rejected'")); ?></div>
                    <div class="stat-label">İptal Edilen</div>
                </div>
            </div>
            
            <div class="clients-filters">
                <form method="get" style="display: flex; gap: 12px; align-items: center; flex-wrap: wrap; width: 100%;">
                    <input type="hidden" name="page" value="appointment-booking-clients">
                    <input type="text" name="s" value="<?php echo esc_attr($search); ?>" placeholder="Ad, soyad veya telefon ara..." class="filter-input">
                    <select name="status" class="filter-select">
                        <option value="">Tüm Durumlar</option>
                        <option value="confirmed" <?php selected($status_filter, 'confirmed'); ?>>Onaylandı</option>
                        <option value="pending" <?php selected($status_filter, 'pending'); ?>>Bekliyor</option>
                        <option value="rejected" <?php selected($status_filter, 'rejected'); ?>>İptal Edildi</option>
                    </select>
                    <button type="submit" class="filter-btn">Filtrele</button>
                    <?php if ($search || $status_filter): ?>
                    <a href="?page=appointment-booking-clients" style="padding: 10px 20px; background: #f3f4f6; color: #374151; border-radius: 8px; font-size: 14px; font-weight: 600; text-decoration: none;">Temizle</a>
                    <?php endif; ?>
                </form>
            </div>
            
            <div class="clients-table">
                <table>
                    <thead>
                        <tr>
                            <th>Danışan</th>
                            <th>Telefon</th>
                            <th>Son Randevu</th>
                            <th>Toplam Randevu</th>
                            <th>Durum</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($clients)): ?>
                        <tr>
                            <td colspan="5" style="text-align: center; padding: 40px; color: #9ca3af;">
                                Henüz danışan kaydı yok.
                            </td>
                        </tr>
                        <?php else: ?>
                        <?php foreach ($clients as $client): ?>
                        <tr>
                            <td>
                                <div class="client-name"><?php echo esc_html($client->user_name . ' ' . $client->user_surname); ?></div>
                            </td>
                            <td>
                                <div class="client-phone"><?php echo esc_html($this->format_phone($client->user_phone)); ?></div>
                            </td>
                            <td>
                                <?php 
                                $months_tr = array('', 'Ocak', 'Şubat', 'Mart', 'Nisan', 'Mayıs', 'Haziran', 'Temmuz', 'Ağustos', 'Eylül', 'Ekim', 'Kasım', 'Aralık');
                                $date_obj = strtotime($client->last_appointment_date);
                                echo date('d', $date_obj) . ' ' . $months_tr[date('n', $date_obj)] . ' ' . date('Y', $date_obj);
                                echo ' - ' . substr($client->last_appointment_time, 0, 5);
                                ?>
                            </td>
                            <td><?php echo $client->total_appointments; ?> randevu</td>
                            <td>
                                <span class="appointment-badge badge-confirmed"><?php echo $client->confirmed_count; ?> Onaylı</span>
                                <span class="appointment-badge badge-pending"><?php echo $client->pending_count; ?> Bekliyor</span>
                                <span class="appointment-badge badge-rejected"><?php echo $client->rejected_count; ?> İptal</span>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
            
            <?php if ($total_pages > 1): ?>
            <div class="pagination">
                <?php if ($current_page > 1): ?>
                <a href="?page=appointment-booking-clients&paged=<?php echo ($current_page - 1); ?><?php echo $search ? '&s=' . urlencode($search) : ''; ?><?php echo $status_filter ? '&status=' . $status_filter : ''; ?>">« Önceki</a>
                <?php endif; ?>
                
                <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                    <?php if ($i == $current_page): ?>
                    <span class="current"><?php echo $i; ?></span>
                    <?php else: ?>
                    <a href="?page=appointment-booking-clients&paged=<?php echo $i; ?><?php echo $search ? '&s=' . urlencode($search) : ''; ?><?php echo $status_filter ? '&status=' . $status_filter : ''; ?>"><?php echo $i; ?></a>
                    <?php endif; ?>
                <?php endfor; ?>
                
                <?php if ($current_page < $total_pages): ?>
                <a href="?page=appointment-booking-clients&paged=<?php echo ($current_page + 1); ?><?php echo $search ? '&s=' . urlencode($search) : ''; ?><?php echo $status_filter ? '&status=' . $status_filter : ''; ?>">Sonraki »</a>
                <?php endif; ?>
            </div>
            <?php endif; ?>
        </div>
        </div>
        <?php
    }
    
    /**
     * AJAX: Admin add appointment
     */
    public function ajax_admin_add_appointment() {
        check_ajax_referer('appointment_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Yetkiniz yok.'));
        }
        
        $data = array(
            'user_name' => isset($_POST['user_name']) ? sanitize_text_field($_POST['user_name']) : '',
            'user_surname' => isset($_POST['user_surname']) ? sanitize_text_field($_POST['user_surname']) : '',
            'user_phone' => isset($_POST['user_phone']) ? sanitize_text_field($_POST['user_phone']) : '',
            'appointment_date' => isset($_POST['appointment_date']) ? sanitize_text_field($_POST['appointment_date']) : '',
            'appointment_time' => isset($_POST['appointment_time']) ? sanitize_text_field($_POST['appointment_time']) : '',
            'appointment_end_time' => isset($_POST['appointment_end_time']) ? sanitize_text_field($_POST['appointment_end_time']) : '',
            'status' => isset($_POST['status']) ? sanitize_text_field($_POST['status']) : 'pending'
        );
        
        // Validate required fields
        if (empty($data['user_name']) || empty($data['user_surname']) || empty($data['user_phone']) || empty($data['appointment_date']) || empty($data['appointment_time'])) {
            wp_send_json_error(array('message' => 'Lütfen tüm zorunlu alanları doldurun.'));
        }
        
        global $wpdb;
        $appointments_table = $wpdb->prefix . 'appointments';
        $blocked_table = $wpdb->prefix . 'appointment_blocked_dates';
        
        // Check if date is blocked
        $blocked = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$blocked_table} WHERE blocked_date = %s",
            $data['appointment_date']
        ));
        
        if ($blocked) {
            // Check if specific time is blocked
            if ($blocked->start_time && $blocked->end_time) {
                $appointment_time = strtotime($data['appointment_time']);
                $block_start = strtotime($blocked->start_time);
                $block_end = strtotime($blocked->end_time);
                
                if ($appointment_time >= $block_start && $appointment_time < $block_end) {
                    wp_send_json_error(array('message' => 'Bu tarih veya saat blokeli. Lütfen başka bir zaman seçin.'));
                }
            } else {
                // Entire day is blocked
                wp_send_json_error(array('message' => 'Bu tarih tamamen blokeli. Lütfen başka bir tarih seçin.'));
            }
        }
        
        // Check if time slot is already taken
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$appointments_table} 
            WHERE appointment_date = %s 
            AND appointment_time = %s 
            AND status IN ('pending', 'confirmed')",
            $data['appointment_date'],
            $data['appointment_time']
        ));
        
        if ($existing > 0) {
            wp_send_json_error(array('message' => 'Bu tarih ve saatte zaten bir randevu var. Lütfen başka bir saat seçin.'));
        }
        
        // Insert appointment
        $insert_data = array(
            'user_name' => $data['user_name'],
            'user_surname' => $data['user_surname'],
            'user_phone' => $data['user_phone'],
            'appointment_date' => $data['appointment_date'],
            'appointment_time' => $data['appointment_time'],
            'status' => $data['status'],
            'created_at' => current_time('mysql')
        );
        
        $format = array('%s', '%s', '%s', '%s', '%s', '%s', '%s');
        
        // Add end time if provided
        if (!empty($data['appointment_end_time'])) {
            $insert_data['appointment_end_time'] = $data['appointment_end_time'];
            $format[] = '%s';
        }
        
        $result = $wpdb->insert($appointments_table, $insert_data, $format);
        
        if ($result) {
            wp_send_json_success(array('message' => 'Randevu başarıyla eklendi!'));
        } else {
            wp_send_json_error(array('message' => 'Veritabanı hatası oluştu. Lütfen tekrar deneyin.'));
        }
    }

    /**
     * Handle admin add appointment
     */
    public function handle_admin_add_appointment() {
        check_admin_referer('appointment_admin_add_nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die(__('Yetkiniz yok.', 'appointment-booking'));
        }
        
        $data = array(
            'user_name' => isset($_POST['user_name']) ? sanitize_text_field($_POST['user_name']) : '',
            'user_surname' => isset($_POST['user_surname']) ? sanitize_text_field($_POST['user_surname']) : '',
            'user_phone' => isset($_POST['user_phone']) ? sanitize_text_field($_POST['user_phone']) : '',
            'appointment_date' => isset($_POST['appointment_date']) ? sanitize_text_field($_POST['appointment_date']) : '',
            'appointment_time' => isset($_POST['appointment_time']) ? sanitize_text_field($_POST['appointment_time']) : '',
            'appointment_end_time' => isset($_POST['appointment_end_time']) ? sanitize_text_field($_POST['appointment_end_time']) : '',
            'status' => isset($_POST['status']) ? sanitize_text_field($_POST['status']) : 'pending'
        );
        
        // Validate required fields
        if (empty($data['user_name']) || empty($data['user_surname']) || empty($data['user_phone']) || empty($data['appointment_date']) || empty($data['appointment_time'])) {
            wp_redirect(admin_url('admin.php?page=appointment-booking&error=missing_fields'));
            exit;
        }
        
        global $wpdb;
        $appointments_table = $wpdb->prefix . 'appointments';
        $blocked_table = $wpdb->prefix . 'appointment_blocked_dates';
        
        // Check if date is blocked
        $blocked = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$blocked_table} WHERE blocked_date = %s",
            $data['appointment_date']
        ));
        
        if ($blocked) {
            // Check if specific time is blocked
            if ($blocked->start_time && $blocked->end_time) {
                $appointment_time = strtotime($data['appointment_time']);
                $block_start = strtotime($blocked->start_time);
                $block_end = strtotime($blocked->end_time);
                
                if ($appointment_time >= $block_start && $appointment_time < $block_end) {
                    wp_redirect(admin_url('admin.php?page=appointment-booking&error=date_blocked'));
                    exit;
                }
            } else {
                // Entire day is blocked
                wp_redirect(admin_url('admin.php?page=appointment-booking&error=date_blocked'));
                exit;
            }
        }
        
        // Check if time slot is already taken
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$appointments_table} 
            WHERE appointment_date = %s 
            AND appointment_time = %s 
            AND status IN ('pending', 'confirmed')",
            $data['appointment_date'],
            $data['appointment_time']
        ));
        
        if ($existing > 0) {
            wp_redirect(admin_url('admin.php?page=appointment-booking&error=time_taken'));
            exit;
        }
        
        // Insert appointment
        $insert_data = array(
            'user_name' => $data['user_name'],
            'user_surname' => $data['user_surname'],
            'user_phone' => $data['user_phone'],
            'appointment_date' => $data['appointment_date'],
            'appointment_time' => $data['appointment_time'],
            'status' => $data['status'],
            'created_at' => current_time('mysql')
        );
        
        $format = array('%s', '%s', '%s', '%s', '%s', '%s', '%s');
        
        // Add end time if provided
        if (!empty($data['appointment_end_time'])) {
            $insert_data['appointment_end_time'] = $data['appointment_end_time'];
            $format[] = '%s';
        }
        
        $result = $wpdb->insert($appointments_table, $insert_data, $format);
        
        if ($result) {
            wp_redirect(admin_url('admin.php?page=appointment-booking&added=1'));
        } else {
            wp_redirect(admin_url('admin.php?page=appointment-booking&error=db_error'));
        }
        exit;
    }

}
