<?php
/**
 * Backlink protection class
 * Protected with integrity verification
 *
 * @package Appointment_Booking
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Appointment_Booking_Backlink_Guard {
    
    // Encoded domain parts - DO NOT MODIFY
    private $_d1 = 'cHNpa2l5YXRyaXJlaGJlcmk='; // base64
    private $_d2 = 'Y29tLnRy'; // base64
    private $_p = 'aHR0cHM6Ly8='; // https://
    private $_s = 'Lw=='; // /
    private $_t = 'UHNpa2l5YXRyaSBSZWhiZXJp'; // backlink text encoded (Psikiyatri Rehberi)
    
    // Integrity hash - generated from original values
    private $_ih = 'f41ee21836775b3e';
    
    /**
     * Get backlink text
     */
    private function get_text() {
        return base64_decode($this->_t);
    }
    
    /**
     * Get the protected URL
     */
    private function get_url() {
        return base64_decode($this->_p) . base64_decode($this->_d1) . '.' . base64_decode($this->_d2) . base64_decode($this->_s);
    }
    
    /**
     * Get domain for verification
     */
    private function get_domain() {
        return base64_decode($this->_d1) . '.' . base64_decode($this->_d2);
    }
    
    /**
     * Verify integrity - check if encoded values match expected hash
     */
    private function verify_integrity() {
        $check = md5($this->_d1 . $this->_d2 . $this->_p . $this->_s . 'apb2024');
        return substr($check, 0, 16) === $this->_ih;
    }
    
    /**
     * Cross-verify with main license URL
     */
    private function verify_license_match() {
        if (!function_exists('appointment_booking_get_license_domain')) {
            return false;
        }
        return appointment_booking_get_license_domain() === $this->get_domain();
    }
    
    /**
     * Verify backlink presence
     *
     * @return bool
     */
    public function verify() {
        // Verify integrity of encoded values
        if (!$this->verify_integrity()) {
            return false;
        }
        
        // Cross-verify with main license
        if (!$this->verify_license_match()) {
            return false;
        }
        
        return true;
    }
    
    /**
     * Render backlink HTML
     *
     * @return string
     */
    public function render_link() {
        // Verify before rendering
        if (!$this->verify()) {
            return '<div class="appointment-error" style="padding:20px;background:#fee2e2;border:1px solid #fecaca;border-radius:8px;color:#991b1b;text-align:center;">Eklenti bütünlük hatası.</div>';
        }
        
        $url = $this->get_url();
        $text = $this->get_text();
        $verification_hash = $this->generate_verification_hash();
        
        $link_html = sprintf(
            '<a href="%s" target="_blank" rel="nofollow" class="appointment-backlink" data-v="%s">%s</a>',
            esc_url($url),
            esc_attr($verification_hash),
            esc_html($text)
        );
        
        $output = '<div class="appointment-footer" id="appointment-footer-link">';
        $output .= $link_html;
        $output .= '</div>';
        
        // Add JavaScript protection with encoded verification
        $output .= $this->get_protection_script();
        
        return $output;
    }
    
    /**
     * Generate verification hash for JavaScript
     */
    private function generate_verification_hash() {
        $content = $this->get_url() . $this->get_text() . date('Y-m');
        return substr(md5($content), 0, 12);
    }
    
    /**
     * Get JavaScript protection script
     */
    private function get_protection_script() {
        // Encode URL parts for JavaScript
        $js_parts = array(
            'a' => base64_encode('https://'),
            'b' => $this->_d1,
            'c' => base64_encode('.'),
            'd' => $this->_d2,
            'e' => base64_encode('/')
        );
        $expected_hash = $this->generate_verification_hash();
        
        ob_start();
        ?>
        <script>
        (function() {
            'use strict';
            var _p = <?php echo json_encode($js_parts); ?>;
            var _h = '<?php echo esc_js($expected_hash); ?>';
            
            function _d(s) { return atob(s); }
            function _u() { return _d(_p.a) + _d(_p.b) + _d(_p.c) + _d(_p.d) + _d(_p.e); }
            
            function _c() {
                var f = document.getElementById('appointment-footer-link');
                if (!f) { _x(); return; }
                
                var l = f.querySelector('.appointment-backlink');
                if (!l) { _x(); return; }
                
                var s = window.getComputedStyle(f);
                if (s.display === 'none' || s.visibility === 'hidden' || parseFloat(s.opacity) < 0.1 || f.offsetHeight < 5) {
                    _x(); return;
                }
                
                var h = l.getAttribute('href');
                var eu = _u();
                if (h !== eu) { _x(); return; }
                
                var v = l.getAttribute('data-v');
                if (v !== _h) { _x(); return; }
            }
            
            function _x() {
                var w = document.querySelectorAll('.appointment-booking-wrapper');
                w.forEach(function(e) {
                    e.innerHTML = '<div style="padding:20px;background:#fee2e2;border:1px solid #fecaca;border-radius:8px;color:#991b1b;text-align:center;">Eklenti lisans hatası nedeniyle çalışmıyor.</div>';
                });
            }
            
            if (document.readyState === 'loading') {
                document.addEventListener('DOMContentLoaded', _c);
            } else {
                setTimeout(_c, 100);
            }
            
            setInterval(_c, 3000);
            
            var o = new MutationObserver(function() { _c(); });
            if (document.body) {
                o.observe(document.body, { childList: true, subtree: true, attributes: true, attributeFilter: ['style', 'class', 'href'] });
            }
        })();
        </script>
        <?php
        return ob_get_clean();
    }
}
