<?php
/**
 * Calendar logic class
 *
 * @package Appointment_Booking
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Appointment_Booking_Calendar {
    
    /**
     * @var Appointment_Booking_DB
     */
    private $db;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->db = new Appointment_Booking_DB();
    }
    
    /**
     * Generate calendar data for a month
     *
     * @param int $year
     * @param int $month
     * @return array
     */
    public function generate_calendar($year, $month) {
        $year = absint($year);
        $month = absint($month);
        
        $first_day = mktime(0, 0, 0, $month, 1, $year);
        $days_in_month = date('t', $first_day);
        $day_of_week = date('w', $first_day);
        
        // Convert Sunday=0 to Monday=0 format (ISO-8601)
        // Sunday should be 6, Monday should be 0
        $first_day_of_week = ($day_of_week == 0) ? 6 : $day_of_week - 1;
        
        $calendar = array(
            'year' => $year,
            'month' => $month,
            'month_name' => date('F', $first_day),
            'days_in_month' => $days_in_month,
            'first_day_of_week' => $first_day_of_week,
            'days' => array()
        );
        
        // Get available dates for this month
        $available_dates = $this->get_available_dates_for_month($year, $month);
        
        for ($day = 1; $day <= $days_in_month; $day++) {
            $date = sprintf('%04d-%02d-%02d', $year, $month, $day);
            $timestamp = strtotime($date);
            
            $calendar['days'][] = array(
                'day' => $day,
                'date' => $date,
                'day_of_week' => date('w', $timestamp),
                'is_available' => in_array($date, $available_dates),
                'is_past' => $timestamp < strtotime('today'),
                'is_today' => $date === date('Y-m-d')
            );
        }
        
        return $calendar;
    }
    
    /**
     * Get available dates for a month
     *
     * @param int $year
     * @param int $month
     * @return array
     */
    public function get_available_dates_for_month($year, $month) {
        $year = absint($year);
        $month = absint($month);
        
        $available_dates = array();
        $days_in_month = cal_days_in_month(CAL_GREGORIAN, $month, $year);
        $today = strtotime('today');
        
        // Get blocked dates
        $blocked_dates = $this->db->get_blocked_dates();
        
        for ($day = 1; $day <= $days_in_month; $day++) {
            $date = sprintf('%04d-%02d-%02d', $year, $month, $day);
            $timestamp = strtotime($date);
            
            // Skip past dates
            if ($timestamp < $today) {
                continue;
            }
            
            // Skip blocked dates
            if (in_array($date, $blocked_dates)) {
                continue;
            }
            
            // Check if date is available
            if ($this->is_date_available($date)) {
                $available_dates[] = $date;
            }
        }
        
        return $available_dates;
    }
    
    /**
     * Get time slots for a specific date
     *
     * @param string $date
     * @return array
     */
    public function get_time_slots_for_date($date) {
        $date = sanitize_text_field($date);
        
        // Check if date is in the past
        if (strtotime($date) < strtotime('today')) {
            return array();
        }
        
        // Check if date is blocked
        $blocked_dates = $this->db->get_blocked_dates();
        if (in_array($date, $blocked_dates)) {
            return array();
        }
        
        // Get day of week
        $day_of_week = date('w', strtotime($date));
        
        // Get working hours for this day
        $working_hours = $this->db->get_working_hours($day_of_week);
        
        // If no working hours defined, use defaults for weekdays
        if (empty($working_hours)) {
            // Default: Monday-Friday (1-5), 09:00-17:00, 30 min slots
            if ($day_of_week >= 1 && $day_of_week <= 5) {
                $working_hours = array(
                    (object) array(
                        'start_time' => '09:00:00',
                        'end_time' => '17:00:00',
                        'slot_duration' => 30
                    )
                );
            } else {
                return array();
            }
        }
        
        // Calculate slots
        $slots = $this->calculate_slots($date, $working_hours);
        
        // Filter booked slots
        $slots = $this->filter_booked_slots($date, $slots);
        
        return $slots;
    }
    
    /**
     * Check if a date is available
     *
     * @param string $date
     * @return bool
     */
    public function is_date_available($date) {
        $date = sanitize_text_field($date);
        
        // Check if date is in the past
        if (strtotime($date) < strtotime('today')) {
            return false;
        }
        
        // Check if date is blocked
        $blocked_dates = $this->db->get_blocked_dates();
        if (in_array($date, $blocked_dates)) {
            return false;
        }
        
        // Get day of week
        $day_of_week = date('w', strtotime($date));
        
        // Check if there are working hours for this day
        $working_hours = $this->db->get_working_hours($day_of_week);
        
        // If no working hours defined for this specific day
        if (empty($working_hours)) {
            // Check if ANY working hours are defined in the system
            global $wpdb;
            $slots_table = $wpdb->prefix . 'appointment_slots';
            $has_any_schedule = $wpdb->get_var("SELECT COUNT(*) FROM {$slots_table} WHERE is_active = 1");
            
            if ($has_any_schedule > 0) {
                // Schedule exists but this day is not included - day is closed
                return false;
            } else {
                // No schedule defined at all - use default (all weekdays available)
                // This allows the plugin to work out of the box
                if ($day_of_week >= 1 && $day_of_week <= 5) {
                    return true; // Monday-Friday available by default
                }
                return false; // Weekend closed by default
            }
        }
        
        return true; // Has working hours, so available
    }
    
    /**
     * Calculate time slots based on working hours
     *
     * @param string $date
     * @param array $working_hours
     * @return array
     */
    private function calculate_slots($date, $working_hours) {
        $slots = array();
        
        // Get default duration from settings
        $settings = get_option('appointment_booking_settings', array());
        $default_duration = isset($settings['slot_duration']) ? intval($settings['slot_duration']) : 30;
        
        foreach ($working_hours as $hours) {
            $start = strtotime($date . ' ' . $hours->start_time);
            $end = strtotime($date . ' ' . $hours->end_time);
            
            // Use slot duration from hours, or default
            $slot_duration = isset($hours->slot_duration) && $hours->slot_duration > 0 ? intval($hours->slot_duration) : $default_duration;
            $duration = $slot_duration * 60; // Convert to seconds
            
            // Validate that slots don't overlap
            $current_time = $start;
            while ($current_time < $end) {
                $next_time = $current_time + $duration;
                
                // Make sure we don't exceed end time
                if ($next_time > $end) {
                    break;
                }
                
                $slots[] = array(
                    'time' => date('H:i', $current_time),
                    'time_full' => date('H:i:s', $current_time),
                    'is_available' => true,
                    'duration' => $slot_duration
                );
                
                $current_time = $next_time;
            }
        }
        
        // Remove duplicates and sort
        $unique_slots = array();
        $seen_times = array();
        
        foreach ($slots as $slot) {
            if (!in_array($slot['time'], $seen_times)) {
                $unique_slots[] = $slot;
                $seen_times[] = $slot['time'];
            }
        }
        
        usort($unique_slots, function($a, $b) {
            return strcmp($a['time'], $b['time']);
        });
        
        return $unique_slots;
    }
    
    /**
     * Filter out booked slots
     *
     * @param string $date
     * @param array $slots
     * @return array
     */
    private function filter_booked_slots($date, $slots) {
        foreach ($slots as &$slot) {
            $is_available = $this->db->is_slot_available($date, $slot['time_full']);
            $slot['is_available'] = $is_available;
        }
        
        return $slots;
    }
}
