<?php
/**
 * Database operations class
 *
 * @package Appointment_Booking
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Appointment_Booking_DB {
    
    /**
     * @var wpdb
     */
    private $wpdb;
    
    /**
     * @var string
     */
    private $appointments_table;
    
    /**
     * @var string
     */
    private $slots_table;
    
    /**
     * @var string
     */
    private $blocked_dates_table;
    
    /**
     * Constructor
     */
    public function __construct() {
        global $wpdb;
        $this->wpdb = $wpdb;
        $this->appointments_table = $wpdb->prefix . 'appointments';
        $this->slots_table = $wpdb->prefix . 'appointment_slots';
        $this->blocked_dates_table = $wpdb->prefix . 'appointment_blocked_dates';
    }
    
    /**
     * Create database tables
     *
     * @return bool
     */
    public function create_tables() {
        $charset_collate = $this->wpdb->get_charset_collate();
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        
        // Appointments table
        $sql_appointments = "CREATE TABLE IF NOT EXISTS {$this->appointments_table} (
            id BIGINT(20) UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            user_name VARCHAR(100) NOT NULL,
            user_surname VARCHAR(100) NOT NULL,
            user_phone VARCHAR(20) NOT NULL,
            appointment_date DATE NOT NULL,
            appointment_time TIME NOT NULL,
            appointment_end_time TIME NULL,
            status ENUM('pending', 'confirmed', 'rejected', 'cancelled') DEFAULT 'pending',
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            INDEX idx_date_time (appointment_date, appointment_time),
            INDEX idx_status (status)
        ) $charset_collate;";
        
        // Appointment slots table
        $sql_slots = "CREATE TABLE IF NOT EXISTS {$this->slots_table} (
            id BIGINT(20) UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            day_of_week TINYINT(1) NOT NULL,
            start_time TIME NOT NULL,
            end_time TIME NOT NULL,
            slot_duration INT NOT NULL,
            is_active BOOLEAN DEFAULT TRUE,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_day (day_of_week)
        ) $charset_collate;";
        
        // Blocked dates table
        $sql_blocked = "CREATE TABLE IF NOT EXISTS {$this->blocked_dates_table} (
            id BIGINT(20) UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            blocked_date DATE NOT NULL,
            start_time TIME DEFAULT NULL,
            end_time TIME DEFAULT NULL,
            reason VARCHAR(255),
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_date (blocked_date)
        ) $charset_collate;";
        
        dbDelta($sql_appointments);
        dbDelta($sql_slots);
        dbDelta($sql_blocked);
        
        return true;
    }
    
    /**
     * Insert new appointment
     *
     * @param array $data
     * @return int|false
     */
    public function insert_appointment($data) {
        // Sanitize data
        $sanitized_data = array(
            'user_name' => sanitize_text_field($data['user_name']),
            'user_surname' => sanitize_text_field($data['user_surname']),
            'user_phone' => $this->sanitize_phone($data['user_phone']),
            'appointment_date' => sanitize_text_field($data['appointment_date']),
            'appointment_time' => sanitize_text_field($data['appointment_time']),
            'status' => isset($data['status']) ? sanitize_text_field($data['status']) : 'pending'
        );
        
        $result = $this->wpdb->insert(
            $this->appointments_table,
            $sanitized_data,
            array('%s', '%s', '%s', '%s', '%s', '%s')
        );
        
        if ($result === false) {
            return false;
        }
        
        return $this->wpdb->insert_id;
    }
    
    /**
     * Count pending appointments by phone number
     *
     * @param string $phone
     * @return int
     */
    public function count_pending_by_phone($phone) {
        $phone_clean = preg_replace('/[^0-9]/', '', $phone);
        
        // Get last 10 digits for comparison
        if (strlen($phone_clean) > 10) {
            $phone_clean = substr($phone_clean, -10);
        }
        
        $query = $this->wpdb->prepare(
            "SELECT COUNT(*) FROM {$this->appointments_table} 
             WHERE REPLACE(REPLACE(user_phone, ' ', ''), '-', '') LIKE %s 
             AND status = 'pending'
             AND appointment_date >= CURDATE()",
            '%' . $phone_clean
        );
        
        return (int) $this->wpdb->get_var($query);
    }
    
    /**
     * Get appointment by ID
     *
     * @param int $id
     * @return object|null
     */
    public function get_appointment($id) {
        $id = absint($id);
        
        $query = $this->wpdb->prepare(
            "SELECT * FROM {$this->appointments_table} WHERE id = %d",
            $id
        );
        
        return $this->wpdb->get_row($query);
    }
    
    /**
     * Get appointments with filters
     *
     * @param array $filters
     * @return array
     */
    public function get_appointments($filters = array()) {
        $where = array('1=1');
        $params = array();
        
        if (!empty($filters['status'])) {
            $where[] = 'status = %s';
            $params[] = sanitize_text_field($filters['status']);
        }
        
        if (!empty($filters['date_from'])) {
            $where[] = 'appointment_date >= %s';
            $params[] = sanitize_text_field($filters['date_from']);
        }
        
        if (!empty($filters['date_to'])) {
            $where[] = 'appointment_date <= %s';
            $params[] = sanitize_text_field($filters['date_to']);
        }
        
        if (!empty($filters['search'])) {
            $search = '%' . $this->wpdb->esc_like(sanitize_text_field($filters['search'])) . '%';
            $where[] = '(user_name LIKE %s OR user_surname LIKE %s OR user_phone LIKE %s)';
            $params[] = $search;
            $params[] = $search;
            $params[] = $search;
        }
        
        $order_by = 'ORDER BY appointment_date DESC, appointment_time DESC';
        
        if (!empty($filters['order_by'])) {
            $allowed_orders = array('appointment_date', 'appointment_time', 'status', 'created_at');
            $order_field = sanitize_text_field($filters['order_by']);
            if (in_array($order_field, $allowed_orders)) {
                $order_dir = !empty($filters['order_dir']) && strtoupper($filters['order_dir']) === 'ASC' ? 'ASC' : 'DESC';
                $order_by = "ORDER BY {$order_field} {$order_dir}";
            }
        }
        
        // Pagination
        $limit = '';
        if (isset($filters['per_page']) && isset($filters['page'])) {
            $per_page = absint($filters['per_page']);
            $page = absint($filters['page']);
            $offset = ($page - 1) * $per_page;
            $limit = " LIMIT {$per_page} OFFSET {$offset}";
        }
        
        $where_clause = implode(' AND ', $where);
        $query = "SELECT * FROM {$this->appointments_table} WHERE {$where_clause} {$order_by}{$limit}";
        
        if (!empty($params)) {
            $query = $this->wpdb->prepare($query, $params);
        }
        
        return $this->wpdb->get_results($query);
    }
    
    /**
     * Count appointments with filters
     *
     * @param array $filters
     * @return int
     */
    public function count_appointments($filters = array()) {
        $where = array('1=1');
        $params = array();
        
        if (!empty($filters['status'])) {
            $where[] = 'status = %s';
            $params[] = sanitize_text_field($filters['status']);
        }
        
        if (!empty($filters['date_from'])) {
            $where[] = 'appointment_date >= %s';
            $params[] = sanitize_text_field($filters['date_from']);
        }
        
        if (!empty($filters['date_to'])) {
            $where[] = 'appointment_date <= %s';
            $params[] = sanitize_text_field($filters['date_to']);
        }
        
        if (!empty($filters['search'])) {
            $search = '%' . $this->wpdb->esc_like(sanitize_text_field($filters['search'])) . '%';
            $where[] = '(user_name LIKE %s OR user_surname LIKE %s OR user_phone LIKE %s)';
            $params[] = $search;
            $params[] = $search;
            $params[] = $search;
        }
        
        $where_clause = implode(' AND ', $where);
        $query = "SELECT COUNT(*) FROM {$this->appointments_table} WHERE {$where_clause}";
        
        if (!empty($params)) {
            $query = $this->wpdb->prepare($query, $params);
        }
        
        return (int) $this->wpdb->get_var($query);
    }
    
    /**
     * Update appointment status
     *
     * @param int $id
     * @param string $status
     * @return bool
     */
    public function update_appointment_status($id, $status) {
        $id = absint($id);
        $allowed_statuses = array('pending', 'confirmed', 'rejected', 'cancelled');
        
        if (!in_array($status, $allowed_statuses)) {
            return false;
        }
        
        $result = $this->wpdb->update(
            $this->appointments_table,
            array('status' => $status),
            array('id' => $id),
            array('%s'),
            array('%d')
        );
        
        return $result !== false;
    }
    
    /**
     * Delete appointment
     *
     * @param int $id
     * @return bool
     */
    public function delete_appointment($id) {
        $id = absint($id);
        
        $result = $this->wpdb->delete(
            $this->appointments_table,
            array('id' => $id),
            array('%d')
        );
        
        return $result !== false;
    }
    
    /**
     * Get available slots for a date
     *
     * @param string $date
     * @return array
     */
    public function get_available_slots($date) {
        $date = sanitize_text_field($date);
        $day_of_week = date('w', strtotime($date));
        
        // Get working hours for this day
        $working_hours = $this->get_working_hours($day_of_week);
        
        if (empty($working_hours)) {
            return array();
        }
        
        // Get booked slots
        $query = $this->wpdb->prepare(
            "SELECT appointment_time FROM {$this->appointments_table} 
            WHERE appointment_date = %s AND status IN ('pending', 'confirmed')",
            $date
        );
        
        $booked = $this->wpdb->get_col($query);
        
        // Generate all possible slots
        $slots = array();
        foreach ($working_hours as $hours) {
            $start = strtotime($date . ' ' . $hours->start_time);
            $end = strtotime($date . ' ' . $hours->end_time);
            $duration = $hours->slot_duration * 60; // Convert to seconds
            
            for ($time = $start; $time < $end; $time += $duration) {
                $slot_time = date('H:i:s', $time);
                $slots[] = array(
                    'time' => date('H:i', $time),
                    'is_available' => !in_array($slot_time, $booked)
                );
            }
        }
        
        return $slots;
    }
    
    /**
     * Check if slot is available
     *
     * @param string $date
     * @param string $time
     * @return bool
     */
    public function is_slot_available($date, $time) {
        $date = sanitize_text_field($date);
        $time = sanitize_text_field($time);
        
        // Check if date is blocked
        if ($this->is_date_blocked($date)) {
            return false;
        }
        
        // Check if specific time is blocked
        if ($this->is_time_blocked($date, $time)) {
            return false;
        }
        
        // Check if slot is booked
        $query = $this->wpdb->prepare(
            "SELECT COUNT(*) FROM {$this->appointments_table} 
            WHERE appointment_date = %s AND appointment_time = %s AND status IN ('pending', 'confirmed')",
            $date,
            $time
        );
        
        $count = $this->wpdb->get_var($query);
        
        return $count == 0;
    }
    
    /**
     * Get working hours for a day
     *
     * @param int $day_of_week
     * @return array
     */
    public function get_working_hours($day_of_week) {
        $day_of_week = absint($day_of_week);
        
        $query = $this->wpdb->prepare(
            "SELECT * FROM {$this->slots_table} WHERE day_of_week = %d AND is_active = 1",
            $day_of_week
        );
        
        return $this->wpdb->get_results($query);
    }
    
    /**
     * Save working hours
     *
     * @param array $schedule
     * @return bool
     */
    public function save_working_hours($schedule) {
        // Delete existing schedule
        $this->wpdb->query("DELETE FROM {$this->slots_table}");
        
        // Insert new schedule
        foreach ($schedule as $day => $hours) {
            if (!$hours['is_active']) {
                continue;
            }
            
            $this->wpdb->insert(
                $this->slots_table,
                array(
                    'day_of_week' => absint($day),
                    'start_time' => sanitize_text_field($hours['start_time']),
                    'end_time' => sanitize_text_field($hours['end_time']),
                    'slot_duration' => absint($hours['slot_duration']),
                    'is_active' => 1
                ),
                array('%d', '%s', '%s', '%d', '%d')
            );
        }
        
        return true;
    }
    
    /**
     * Block a date or time range
     *
     * @param string $date
     * @param string $reason
     * @param string $start_time
     * @param string $end_time
     * @return bool
     */
    public function block_date($date, $reason = '', $start_time = null, $end_time = null) {
        $data = array(
            'blocked_date' => sanitize_text_field($date),
            'reason' => sanitize_text_field($reason),
            'start_time' => $start_time,
            'end_time' => $end_time
        );
        
        $format = array('%s', '%s', '%s', '%s');
        
        $result = $this->wpdb->insert(
            $this->blocked_dates_table,
            $data,
            $format
        );
        
        return $result !== false;
    }
    
    /**
     * Get blocked dates with time ranges (only today and future)
     *
     * @return array
     */
    public function get_blocked_dates_detailed() {
        $today = date('Y-m-d');
        $query = $this->wpdb->prepare(
            "SELECT * FROM {$this->blocked_dates_table} WHERE blocked_date >= %s ORDER BY blocked_date ASC",
            $today
        );
        return $this->wpdb->get_results($query);
    }
    
    /**
     * Check if date already has a full day block
     *
     * @param string $date
     * @return bool
     */
    public function has_full_day_block($date) {
        $query = $this->wpdb->prepare(
            "SELECT COUNT(*) FROM {$this->blocked_dates_table} 
            WHERE blocked_date = %s AND start_time IS NULL AND end_time IS NULL",
            $date
        );
        return $this->wpdb->get_var($query) > 0;
    }
    
    /**
     * Check if time range overlaps with existing blocks
     *
     * @param string $date
     * @param string $start_time
     * @param string $end_time
     * @return bool
     */
    public function has_overlapping_block($date, $start_time, $end_time) {
        $query = $this->wpdb->prepare(
            "SELECT COUNT(*) FROM {$this->blocked_dates_table} 
            WHERE blocked_date = %s 
            AND (
                (start_time IS NULL AND end_time IS NULL)
                OR (start_time < %s AND end_time > %s)
                OR (start_time < %s AND end_time > %s)
                OR (start_time >= %s AND end_time <= %s)
            )",
            $date,
            $end_time, $start_time,
            $end_time, $end_time,
            $start_time, $end_time
        );
        return $this->wpdb->get_var($query) > 0;
    }
    
    /**
     * Check if date has any appointments (pending or confirmed)
     *
     * @param string $date
     * @return bool
     */
    public function has_appointments_on_date($date) {
        $query = $this->wpdb->prepare(
            "SELECT COUNT(*) FROM {$this->appointments_table} 
            WHERE appointment_date = %s AND status IN ('pending', 'confirmed')",
            $date
        );
        return $this->wpdb->get_var($query) > 0;
    }
    
    /**
     * Check if time range has any appointments
     *
     * @param string $date
     * @param string $start_time
     * @param string $end_time
     * @return bool
     */
    public function has_appointments_in_range($date, $start_time, $end_time) {
        $query = $this->wpdb->prepare(
            "SELECT COUNT(*) FROM {$this->appointments_table} 
            WHERE appointment_date = %s 
            AND appointment_time >= %s 
            AND appointment_time < %s
            AND status IN ('pending', 'confirmed')",
            $date,
            $start_time,
            $end_time
        );
        return $this->wpdb->get_var($query) > 0;
    }
    
    /**
     * Get appointments count for a date
     *
     * @param string $date
     * @return int
     */
    public function get_appointments_count_on_date($date) {
        $query = $this->wpdb->prepare(
            "SELECT COUNT(*) FROM {$this->appointments_table} 
            WHERE appointment_date = %s AND status IN ('pending', 'confirmed')",
            $date
        );
        return (int) $this->wpdb->get_var($query);
    }
    
    /**
     * Check if specific time is blocked (only time-based blocks, not full day blocks)
     *
     * @param string $date
     * @param string $time
     * @return bool
     */
    public function is_time_blocked($date, $time) {
        $query = $this->wpdb->prepare(
            "SELECT COUNT(*) FROM {$this->blocked_dates_table} 
            WHERE blocked_date = %s 
            AND start_time IS NOT NULL 
            AND end_time IS NOT NULL
            AND start_time <= %s 
            AND end_time > %s",
            $date,
            $time,
            $time
        );
        
        return $this->wpdb->get_var($query) > 0;
    }
    
    /**
     * Delete blocked date by ID
     *
     * @param int $id
     * @return bool
     */
    public function delete_blocked_date($id) {
        $result = $this->wpdb->delete(
            $this->blocked_dates_table,
            array('id' => absint($id)),
            array('%d')
        );
        
        return $result !== false;
    }
    
    /**
     * Unblock a date
     *
     * @param string $date
     * @return bool
     */
    public function unblock_date($date) {
        $result = $this->wpdb->delete(
            $this->blocked_dates_table,
            array('blocked_date' => sanitize_text_field($date)),
            array('%s')
        );
        
        return $result !== false;
    }
    
    /**
     * Get blocked dates (only full day blocks, not partial time blocks)
     *
     * @return array
     */
    public function get_blocked_dates() {
        $query = "SELECT blocked_date FROM {$this->blocked_dates_table} 
                  WHERE start_time IS NULL AND end_time IS NULL";
        return $this->wpdb->get_col($query);
    }
    
    /**
     * Check if date is fully blocked (not partial time blocks)
     *
     * @param string $date
     * @return bool
     */
    private function is_date_blocked($date) {
        $query = $this->wpdb->prepare(
            "SELECT COUNT(*) FROM {$this->blocked_dates_table} 
             WHERE blocked_date = %s 
             AND start_time IS NULL 
             AND end_time IS NULL",
            $date
        );
        
        return $this->wpdb->get_var($query) > 0;
    }
    
    /**
     * Sanitize phone number
     *
     * @param string $phone
     * @return string
     */
    private function sanitize_phone($phone) {
        // Remove all non-numeric characters except +
        $phone = preg_replace('/[^0-9+]/', '', $phone);
        return sanitize_text_field($phone);
    }
    
    /**
     * Clean up past blocked dates
     *
     * @return int Number of deleted rows
     */
    public function cleanup_past_blocked_dates() {
        $yesterday = date('Y-m-d', strtotime('-1 day'));
        return $this->wpdb->query(
            $this->wpdb->prepare(
                "DELETE FROM {$this->blocked_dates_table} WHERE blocked_date < %s",
                $yesterday
            )
        );
    }
}
