<?php
/**
 * Form handling class
 *
 * @package Appointment_Booking
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Appointment_Booking_Form {
    
    /**
     * @var Appointment_Booking_DB
     */
    private $db;
    
    /**
     * @var Appointment_Booking_Calendar
     */
    private $calendar;
    
    /**
     * @var Appointment_Booking_Backlink_Guard
     */
    private $backlink_guard;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->db = new Appointment_Booking_DB();
        $this->calendar = new Appointment_Booking_Calendar();
        
        // Initialize backlink guard if class exists
        if (class_exists('Appointment_Booking_Backlink_Guard')) {
            $this->backlink_guard = new Appointment_Booking_Backlink_Guard();
        }
        
        // Register AJAX handlers
        add_action('wp_ajax_appointment_get_available_dates', array($this, 'ajax_get_available_dates'));
        add_action('wp_ajax_nopriv_appointment_get_available_dates', array($this, 'ajax_get_available_dates'));
        
        add_action('wp_ajax_appointment_get_time_slots', array($this, 'ajax_get_time_slots'));
        add_action('wp_ajax_nopriv_appointment_get_time_slots', array($this, 'ajax_get_time_slots'));
        
        add_action('wp_ajax_appointment_submit', array($this, 'ajax_submit_appointment'));
        add_action('wp_ajax_nopriv_appointment_submit', array($this, 'ajax_submit_appointment'));
    }
    
    /**
     * Render booking form
     *
     * @param array $atts
     * @return string
     */
    public function render_form($atts = array()) {
        // Verify license
        if (function_exists('appointment_booking_is_licensed') && !appointment_booking_is_licensed()) {
            return '<div class="appointment-error" style="padding:20px;background:#fee2e2;border:1px solid #fecaca;border-radius:8px;color:#991b1b;text-align:center;">' . esc_html__('Eklenti lisans doğrulaması başarısız.', 'appointment-booking') . '</div>';
        }
        
        // Verify backlink if guard exists
        if ($this->backlink_guard && !$this->backlink_guard->verify()) {
            return '<div class="appointment-error" style="padding:20px;background:#fee2e2;border:1px solid #fecaca;border-radius:8px;color:#991b1b;text-align:center;">' . esc_html__('Eklenti lisans hatası nedeniyle çalışmıyor.', 'appointment-booking') . '</div>';
        }
        
        $compact = isset($atts['compact']) && $atts['compact'] === 'true';
        
        // Add inline CSS as fallback
        $inline_css = $this->get_inline_css();
        
        ob_start();
        echo '<style>' . $inline_css . '</style>';
        include APPOINTMENT_BOOKING_PLUGIN_DIR . 'templates/booking-form.php';
        return ob_get_clean();
    }
    
    /**
     * Get inline CSS for fallback
     */
    private function get_inline_css() {
        return '
        .appointment-booking-wrapper{max-width:420px;margin:0 auto;background:#fff;border-radius:12px;box-shadow:0 2px 12px rgba(0,0,0,.08);overflow:hidden;font-family:-apple-system,BlinkMacSystemFont,"Segoe UI",Roboto,sans-serif;font-size:14px;color:#333}
        .appointment-booking-wrapper *{box-sizing:border-box;margin:0;padding:0}
        .appointment-title{margin:0;padding:20px 24px;font-size:18px;font-weight:600;color:#fff;text-align:center;background:linear-gradient(135deg,#2563eb 0%,#1d4ed8 100%)}
        .appointment-progress{display:flex;align-items:center;justify-content:center;padding:16px 20px;background:#f8fafc;border-bottom:1px solid #e2e8f0}
        .progress-step{display:flex;flex-direction:column;align-items:center;gap:4px}
        .step-number{width:28px;height:28px;display:flex;align-items:center;justify-content:center;background:#e2e8f0;color:#64748b;border-radius:50%;font-size:12px;font-weight:600}
        .progress-step.active .step-number,.progress-step.completed .step-number{background:#2563eb;color:#fff}
        .step-label{font-size:11px;color:#94a3b8;font-weight:500}
        .progress-step.active .step-label{color:#2563eb}
        .progress-line{width:40px;height:2px;background:#e2e8f0;margin:0 8px;margin-bottom:18px}
        .appointment-step{display:none;padding:20px 24px}
        .appointment-step.active{display:block}
        .appointment-field{margin-bottom:16px}
        .appointment-field label{display:block;margin-bottom:6px;font-weight:500;color:#374151;font-size:13px}
        .appointment-field .required{color:#dc2626}
        .appointment-field input{width:100%;padding:10px 12px;border:1px solid #d1d5db;border-radius:6px;font-size:14px;color:#1f2937;background:#fff}
        .appointment-field input:focus{outline:none;border-color:#2563eb;box-shadow:0 0 0 3px rgba(37,99,235,.1)}
        .appointment-field .field-hint{display:block;margin-top:4px;font-size:11px;color:#9ca3af}
        .appointment-btn{display:inline-flex;align-items:center;justify-content:center;gap:6px;padding:10px 20px;font-size:14px;font-weight:500;border:none;border-radius:6px;cursor:pointer}
        .appointment-btn-next,.appointment-btn-submit{background:#2563eb;color:#fff}
        .appointment-btn-next:hover,.appointment-btn-submit:hover{background:#1d4ed8}
        .appointment-btn-back{background:#f1f5f9;color:#475569}
        .appointment-btn:disabled{background:#e5e7eb;color:#9ca3af;cursor:not-allowed}
        .appointment-actions{display:flex;align-items:center;gap:10px;margin-top:16px}
        .calendar-header{display:flex;justify-content:space-between;align-items:center;margin-bottom:12px}
        .calendar-month-year{font-size:15px;font-weight:600;color:#1f2937}
        .calendar-nav{width:32px;height:32px;display:flex;align-items:center;justify-content:center;background:#f1f5f9;color:#475569;border:none;border-radius:6px;font-size:18px;cursor:pointer}
        .calendar-nav:hover{background:#2563eb;color:#fff}
        .calendar-weekdays{display:grid;grid-template-columns:repeat(7,1fr);gap:4px;margin-bottom:4px}
        .calendar-weekday{text-align:center;font-size:11px;font-weight:600;color:#64748b;padding:8px 0}
        .appointment-calendar{display:grid;grid-template-columns:repeat(7,1fr);gap:4px}
        .calendar-day{aspect-ratio:1;display:flex;align-items:center;justify-content:center;font-size:14px;font-weight:600;color:#1f2937;background:#f8fafc;border:2px solid transparent;border-radius:8px;cursor:pointer;transition:all .15s}
        .calendar-day.empty{background:transparent;cursor:default}
        .calendar-day.available{background:#f0fdf4;color:#166534;border-color:#86efac}
        .calendar-day.available:hover{background:#dcfce7;border-color:#4ade80;transform:scale(1.05)}
        .calendar-day.unavailable,.calendar-day.past{background:#f9fafb;color:#d1d5db;cursor:not-allowed}
        .calendar-day.today{border-color:#2563eb;box-shadow:0 0 0 2px rgba(37,99,235,.2)}
        .calendar-day.selected{background:#2563eb;color:#fff;border-color:#2563eb;transform:scale(1.05)}
        .calendar-loading{grid-column:1/-1;text-align:center;padding:40px 16px;color:#94a3b8;font-size:13px}
        .selected-date-display{display:flex;align-items:center;gap:8px;padding:12px 14px;background:#eff6ff;border-radius:8px;margin-bottom:16px;border:1px solid #bfdbfe}
        .selected-date-display #selected-date-text{color:#1e40af;font-weight:600;font-size:14px}
        .appointment-time-slots{display:grid;grid-template-columns:repeat(3,1fr);gap:8px}
        .time-slot{padding:10px 6px;text-align:center;font-size:13px;font-weight:500;color:#374151;background:#fff;border:1px solid #e5e7eb;border-radius:6px;cursor:pointer}
        .time-slot:hover{border-color:#2563eb;background:#eff6ff}
        .time-slot.selected{background:#2563eb;color:#fff;border-color:#2563eb}
        .time-slot.unavailable{background:#f9fafb;color:#d1d5db;cursor:not-allowed}
        .slots-loading,.no-slots{grid-column:1/-1;text-align:center;padding:30px 16px;color:#94a3b8}
        .appointment-message{padding:12px 14px;border-radius:6px;font-size:13px;font-weight:500;margin:0 24px 16px}
        .appointment-message.success{background:#ecfdf5;color:#065f46;border:1px solid #a7f3d0}
        .appointment-message.error{background:#fef2f2;color:#991b1b;border:1px solid #fecaca}
        .appointment-footer{padding:12px 24px;text-align:center;background:#f8fafc;border-top:1px solid #e2e8f0}
        .appointment-backlink{font-size:11px;color:#94a3b8;text-decoration:none}
        .appointment-form-compact{max-width:100%;border-radius:10px}
        .appointment-form-compact .appointment-title{padding:14px 16px;font-size:15px}
        .appointment-form-compact .appointment-progress{padding:12px 14px}
        .appointment-form-compact .step-number{width:24px;height:24px;font-size:11px}
        .appointment-form-compact .step-label{font-size:10px}
        .appointment-form-compact .progress-line{width:28px;margin-bottom:14px}
        .appointment-form-compact .appointment-step{padding:16px}
        .appointment-form-compact .appointment-field{margin-bottom:12px}
        .appointment-form-compact .appointment-field label{font-size:12px;margin-bottom:4px}
        .appointment-form-compact .appointment-field input{padding:8px 10px;font-size:13px}
        .appointment-form-compact .appointment-btn{padding:8px 14px;font-size:13px}
        .appointment-form-compact .calendar-month-year{font-size:13px}
        .appointment-form-compact .calendar-nav{width:28px;height:28px;font-size:16px}
        .appointment-form-compact .calendar-weekday{font-size:10px;padding:4px 0}
        .appointment-form-compact .calendar-day{font-size:12px;border-radius:6px}
        .appointment-form-compact .time-slot{padding:8px 4px;font-size:12px}
        .appointment-form-compact .selected-date-display{padding:10px 12px;margin-bottom:12px}
        .appointment-form-compact .appointment-footer{padding:10px 16px}
        ';
    }
    
    /**
     * Handle form submission
     *
     * @return array
     */
    public function handle_submission() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'appointment_booking_nonce')) {
            return array(
                'success' => false,
                'message' => __('Güvenlik doğrulaması başarısız.', 'appointment-booking')
            );
        }
        
        // Simple math captcha check
        $captcha_answer = isset($_POST['captcha_answer']) ? intval($_POST['captcha_answer']) : -1;
        $captcha_num1 = isset($_POST['captcha_num1']) ? intval($_POST['captcha_num1']) : 0;
        $captcha_num2 = isset($_POST['captcha_num2']) ? intval($_POST['captcha_num2']) : 0;
        
        if ($captcha_answer !== ($captcha_num1 + $captcha_num2)) {
            return array(
                'success' => false,
                'message' => __('Doğrulama sorusu yanlış. Lütfen tekrar deneyin.', 'appointment-booking')
            );
        }
        
        // Get and validate data
        $data = array(
            'user_name' => isset($_POST['user_name']) ? $_POST['user_name'] : '',
            'user_surname' => isset($_POST['user_surname']) ? $_POST['user_surname'] : '',
            'user_phone' => isset($_POST['user_phone']) ? $_POST['user_phone'] : '',
            'appointment_date' => isset($_POST['appointment_date']) ? $_POST['appointment_date'] : '',
            'appointment_time' => isset($_POST['appointment_time']) ? $_POST['appointment_time'] : ''
        );
        
        $validation = $this->validate_user_data($data);
        
        if (!$validation['valid']) {
            return array(
                'success' => false,
                'message' => implode('<br>', $validation['errors'])
            );
        }
        
        // Check duplicate phone - max 2 pending appointments per phone
        $phone_clean = preg_replace('/[^0-9]/', '', $data['user_phone']);
        $pending_count = $this->db->count_pending_by_phone($phone_clean);
        if ($pending_count >= 2) {
            return array(
                'success' => false,
                'message' => __('Bu telefon numarası ile zaten bekleyen randevunuz var. Lütfen mevcut randevunuzun onaylanmasını bekleyin.', 'appointment-booking')
            );
        }
        
        // Check if slot is still available
        if (!$this->db->is_slot_available($data['appointment_date'], $data['appointment_time'] . ':00')) {
            return array(
                'success' => false,
                'message' => __('Seçilen saat artık müsait değil. Lütfen başka bir saat seçin.', 'appointment-booking')
            );
        }
        
        // Insert appointment
        $appointment_id = $this->db->insert_appointment($data);
        
        if ($appointment_id) {
            // Send email notification
            $this->send_appointment_notification($data, $appointment_id);
            
            return array(
                'success' => true,
                'message' => __('Randevunuz başarıyla kaydedildi! En kısa sürede onay için iletişime geçilecektir.', 'appointment-booking'),
                'appointment_id' => $appointment_id
            );
        } else {
            return array(
                'success' => false,
                'message' => __('Randevu kaydedilemedi. Lütfen tekrar deneyin.', 'appointment-booking')
            );
        }
    }
    
    /**
     * Send email notification for new appointment
     *
     * @param array $data
     * @param int $appointment_id
     */
    private function send_appointment_notification($data, $appointment_id) {
        $email_settings = get_option('appointment_booking_email_settings', array());
        
        // Check if notifications are enabled
        if (!isset($email_settings['enabled']) || !$email_settings['enabled']) {
            return;
        }
        
        $to = isset($email_settings['admin_email']) && $email_settings['admin_email'] ? $email_settings['admin_email'] : get_option('admin_email');
        $from_name = isset($email_settings['from_name']) && $email_settings['from_name'] ? $email_settings['from_name'] : get_bloginfo('name');
        $from_email = isset($email_settings['from_email']) && $email_settings['from_email'] ? $email_settings['from_email'] : get_option('admin_email');
        
        // Format date
        $months_tr = array('', 'Ocak', 'Şubat', 'Mart', 'Nisan', 'Mayıs', 'Haziran', 'Temmuz', 'Ağustos', 'Eylül', 'Ekim', 'Kasım', 'Aralık');
        $date_obj = strtotime($data['appointment_date']);
        $formatted_date = date('d', $date_obj) . ' ' . $months_tr[date('n', $date_obj)] . ' ' . date('Y', $date_obj);
        
        $subject = '🗓️ Yeni Randevu Talebi - ' . $data['user_name'] . ' ' . $data['user_surname'];
        
        $message = "Yeni bir randevu talebi alındı!\n\n";
        $message .= "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
        $message .= "👤 Hasta Bilgileri:\n";
        $message .= "   Ad Soyad: " . $data['user_name'] . " " . $data['user_surname'] . "\n";
        $message .= "   Telefon: " . $data['user_phone'] . "\n\n";
        $message .= "📅 Randevu Bilgileri:\n";
        $message .= "   Tarih: " . $formatted_date . "\n";
        $message .= "   Saat: " . $data['appointment_time'] . "\n\n";
        $message .= "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
        $message .= "Randevuyu onaylamak veya reddetmek için yönetim panelini ziyaret edin:\n";
        $message .= admin_url('admin.php?page=appointment-booking') . "\n\n";
        $message .= "---\n";
        $message .= "Bu e-posta Randevu Eklentisi tarafından otomatik olarak gönderilmiştir.";
        
        $headers = array(
            'Content-Type: text/plain; charset=UTF-8',
            'From: ' . $from_name . ' <' . $from_email . '>'
        );
        
        // If SMTP enabled, configure PHPMailer
        if (isset($email_settings['smtp_enabled']) && $email_settings['smtp_enabled']) {
            add_action('phpmailer_init', function($phpmailer) use ($email_settings) {
                $phpmailer->isSMTP();
                $phpmailer->Host = $email_settings['smtp_host'];
                $phpmailer->Port = $email_settings['smtp_port'];
                $phpmailer->SMTPAuth = true;
                $phpmailer->Username = $email_settings['smtp_user'];
                $phpmailer->Password = $email_settings['smtp_pass'];
                $phpmailer->SMTPSecure = $email_settings['smtp_secure'];
            });
        }
        
        wp_mail($to, $subject, $message, $headers);
    }
    
    /**
     * Validate user data
     *
     * @param array $data
     * @return array
     */
    private function validate_user_data($data) {
        $errors = array();
        
        // Validate name
        if (empty($data['user_name']) || trim($data['user_name']) === '') {
            $errors[] = __('Ad alanı zorunludur.', 'appointment-booking');
        }
        
        // Validate surname
        if (empty($data['user_surname']) || trim($data['user_surname']) === '') {
            $errors[] = __('Soyad alanı zorunludur.', 'appointment-booking');
        }
        
        // Validate phone
        if (empty($data['user_phone'])) {
            $errors[] = __('Telefon alanı zorunludur.', 'appointment-booking');
        } else {
            $phone = $this->sanitize_phone($data['user_phone']);
            if (!$this->is_valid_phone($phone)) {
                $errors[] = __('Geçersiz telefon numarası. Lütfen geçerli bir Türkiye telefon numarası girin.', 'appointment-booking');
            }
        }
        
        // Validate date
        if (empty($data['appointment_date'])) {
            $errors[] = __('Randevu tarihi seçilmedi.', 'appointment-booking');
        } else if (strtotime($data['appointment_date']) < strtotime('today')) {
            $errors[] = __('Geçmiş bir tarih seçilemez.', 'appointment-booking');
        }
        
        // Validate time
        if (empty($data['appointment_time'])) {
            $errors[] = __('Randevu saati seçilmedi.', 'appointment-booking');
        }
        
        return array(
            'valid' => empty($errors),
            'errors' => $errors
        );
    }
    
    /**
     * Sanitize phone number
     *
     * @param string $phone
     * @return string
     */
    private function sanitize_phone($phone) {
        return preg_replace('/[^0-9+]/', '', $phone);
    }
    
    /**
     * Validate Turkish phone number (mobile and landline)
     *
     * @param string $phone
     * @return bool
     */
    private function is_valid_phone($phone) {
        // Remove leading 0 if present
        if (strpos($phone, '0') === 0) {
            $phone = substr($phone, 1);
        }
        
        // Remove country code if present
        if (strpos($phone, '90') === 0) {
            $phone = substr($phone, 2);
        }
        
        // Turkish mobile: 5XXXXXXXXX (10 digits starting with 5)
        if (preg_match('/^5[0-9]{9}$/', $phone)) {
            return true;
        }
        
        // Turkish landline: Area codes starting with 2, 3, 4 (e.g., 212, 312, 232, 416)
        // Format: AAXXXXXXX (10 digits, area code + 7 digit number)
        if (preg_match('/^[234][0-9]{9}$/', $phone)) {
            return true;
        }
        
        return false;
    }
    
    /**
     * AJAX: Get available dates
     */
    public function ajax_get_available_dates() {
        check_ajax_referer('appointment_booking_nonce', 'nonce');
        
        $year = isset($_POST['year']) ? absint($_POST['year']) : date('Y');
        $month = isset($_POST['month']) ? absint($_POST['month']) : date('n');
        
        $calendar = $this->calendar->generate_calendar($year, $month);
        
        wp_send_json_success($calendar);
    }
    
    /**
     * AJAX: Get time slots for a date
     */
    public function ajax_get_time_slots() {
        check_ajax_referer('appointment_booking_nonce', 'nonce');
        
        $date = isset($_POST['date']) ? sanitize_text_field($_POST['date']) : '';
        
        if (empty($date)) {
            wp_send_json_error(array('message' => __('Tarih belirtilmedi.', 'appointment-booking')));
        }
        
        $slots = $this->calendar->get_time_slots_for_date($date);
        
        wp_send_json_success(array('slots' => $slots));
    }
    
    /**
     * AJAX: Submit appointment
     */
    public function ajax_submit_appointment() {
        $result = $this->handle_submission();
        
        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }
}
