<?php
/**
 * Admin dashboard template
 *
 * @package Appointment_Booking
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}
?>
<style>
/* Admin Panel Styles */
.appointment-admin-wrap {
    margin: 20px 20px 20px 0;
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
}
.admin-page-header {
    display: flex;
    align-items: center;
    justify-content: space-between;
    margin-bottom: 24px;
    padding-bottom: 16px;
    border-bottom: 1px solid #e5e7eb;
}
.admin-page-title {
    display: flex;
    align-items: center;
    gap: 12px;
    margin: 0;
    font-size: 24px;
    font-weight: 600;
    color: #111827;
}
.admin-page-title .dashicons {
    font-size: 28px;
    width: 28px;
    height: 28px;
    color: #3b82f6;
}
.btn-add-appointment {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 12px 24px;
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    color: #fff;
    border: none;
    border-radius: 10px;
    font-size: 14px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.2s;
    box-shadow: 0 4px 12px rgba(16,185,129,0.3);
}
.btn-add-appointment:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 16px rgba(16,185,129,0.4);
}
.btn-add-appointment .dashicons {
    font-size: 18px;
    width: 18px;
    height: 18px;
}
/* Stats Grid */
.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 16px;
    margin-bottom: 24px;
}
.stat-card {
    background: #fff;
    border: 1px solid #e5e7eb;
    border-radius: 12px;
    padding: 20px;
    display: flex;
    align-items: center;
    gap: 16px;
    transition: all 0.2s;
}
.stat-card:hover {
    box-shadow: 0 4px 6px -1px rgba(0,0,0,0.1);
    transform: translateY(-2px);
}
.stat-icon {
    width: 48px;
    height: 48px;
    border-radius: 8px;
    display: flex;
    align-items: center;
    justify-content: center;
}
.stat-icon .dashicons {
    font-size: 24px;
    width: 24px;
    height: 24px;
}
.stat-card.total .stat-icon { background: #eff6ff; color: #3b82f6; }
.stat-card.pending .stat-icon { background: #fffbeb; color: #f59e0b; }
.stat-card.confirmed .stat-icon { background: #ecfdf5; color: #10b981; }
.stat-card.rejected .stat-icon { background: #fef2f2; color: #ef4444; }
.stat-number {
    font-size: 28px;
    font-weight: 700;
    color: #111827;
    line-height: 1;
    margin-bottom: 4px;
}
.stat-label {
    font-size: 13px;
    color: #6b7280;
    font-weight: 500;
}
/* Filter Bar - Premium Design */
.filter-bar {
    background: linear-gradient(135deg, #ffffff 0%, #f8fafc 100%);
    border: 1px solid #e2e8f0;
    border-radius: 16px;
    padding: 20px 24px;
    margin-bottom: 24px;
    box-shadow: 0 1px 3px rgba(0,0,0,0.04);
}
.filter-form {
    display: flex;
    flex-direction: column;
    gap: 16px;
}
.filter-row {
    display: flex;
    flex-wrap: wrap;
    gap: 16px;
    align-items: center;
}
.filter-group {
    display: flex;
    align-items: center;
    gap: 8px;
}
.filter-icon-wrapper {
    width: 40px;
    height: 40px;
    background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%);
    border-radius: 10px;
    display: flex;
    align-items: center;
    justify-content: center;
    color: #fff;
}
.filter-icon-wrapper .dashicons {
    font-size: 18px;
    width: 18px;
    height: 18px;
}
.filter-select-styled {
    padding: 10px 36px 10px 14px;
    border: 2px solid #e5e7eb;
    border-radius: 10px;
    font-size: 14px;
    font-weight: 500;
    color: #1f2937;
    background: #fff;
    cursor: pointer;
    min-width: 160px;
    appearance: none;
    background-image: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="%236b7280" stroke-width="2"><polyline points="6 9 12 15 18 9"/></svg>');
    background-repeat: no-repeat;
    background-position: right 12px center;
    transition: all 0.2s;
}
.filter-select-styled:hover {
    border-color: #3b82f6;
}
.filter-select-styled:focus {
    outline: none;
    border-color: #3b82f6;
    box-shadow: 0 0 0 3px rgba(59,130,246,0.1);
}
.date-group {
    background: #fff;
    border: 2px solid #e5e7eb;
    border-radius: 10px;
    padding: 6px 12px;
    transition: all 0.2s;
}
.date-group:hover,
.date-group:focus-within {
    border-color: #3b82f6;
}
.date-group .dashicons {
    color: #3b82f6;
    font-size: 18px;
    width: 18px;
    height: 18px;
}
.filter-date {
    border: none;
    padding: 6px 8px;
    font-size: 14px;
    font-weight: 500;
    color: #1f2937;
    background: transparent;
    width: 100px;
    cursor: pointer;
}
.filter-date:focus {
    outline: none;
}
.filter-date::placeholder {
    color: #9ca3af;
}
.date-sep {
    color: #d1d5db;
    font-size: 12px;
}
.search-group {
    flex: 1;
    min-width: 250px;
    background: #fff;
    border: 2px solid #e5e7eb;
    border-radius: 10px;
    padding: 6px 12px;
    transition: all 0.2s;
}
.search-group:hover,
.search-group:focus-within {
    border-color: #3b82f6;
}
.search-group .dashicons {
    color: #9ca3af;
    font-size: 18px;
    width: 18px;
    height: 18px;
}
.filter-search-input {
    border: none;
    padding: 6px 8px;
    font-size: 14px;
    color: #1f2937;
    background: transparent;
    flex: 1;
    width: 100%;
}
.filter-search-input:focus {
    outline: none;
}
.filter-search-input::placeholder {
    color: #9ca3af;
}
.filter-actions {
    display: flex;
    gap: 8px;
    margin-left: auto;
}
.btn-filter {
    display: inline-flex;
    align-items: center;
    gap: 6px;
    padding: 10px 20px;
    background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%);
    color: #fff;
    border: none;
    border-radius: 10px;
    font-size: 14px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.2s;
    box-shadow: 0 2px 8px rgba(59,130,246,0.3);
}
.btn-filter:hover {
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(59,130,246,0.4);
}
.btn-filter .dashicons {
    font-size: 16px;
    width: 16px;
    height: 16px;
}
.btn-clear {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    width: 40px;
    height: 40px;
    background: #f3f4f6;
    color: #6b7280;
    border: 2px solid #e5e7eb;
    border-radius: 10px;
    text-decoration: none;
    transition: all 0.2s;
}
.btn-clear:hover {
    background: #fef2f2;
    border-color: #fecaca;
    color: #ef4444;
}
.btn-clear .dashicons {
    font-size: 18px;
    width: 18px;
    height: 18px;
}
@media (max-width: 900px) {
    .filter-row {
        flex-direction: column;
        align-items: stretch;
    }
    .filter-group {
        width: 100%;
    }
    .filter-select-styled {
        width: 100%;
    }
    .search-group {
        min-width: auto;
    }
    .filter-actions {
        margin-left: 0;
        justify-content: flex-end;
    }
}
/* Flatpickr Custom Theme */
.flatpickr-calendar {
    border-radius: 12px !important;
    box-shadow: 0 10px 40px rgba(0,0,0,0.15) !important;
    border: none !important;
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif !important;
}
.flatpickr-months {
    background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%) !important;
    border-radius: 12px 12px 0 0 !important;
    padding: 8px 0 !important;
}
.flatpickr-months .flatpickr-month {
    color: #fff !important;
    fill: #fff !important;
}
.flatpickr-current-month {
    font-size: 15px !important;
    font-weight: 600 !important;
}
.flatpickr-current-month .flatpickr-monthDropdown-months {
    background: transparent !important;
    color: #fff !important;
    font-weight: 600 !important;
}
.flatpickr-current-month input.cur-year {
    color: #fff !important;
    font-weight: 600 !important;
}
.flatpickr-months .flatpickr-prev-month,
.flatpickr-months .flatpickr-next-month {
    fill: #fff !important;
    color: #fff !important;
}
.flatpickr-months .flatpickr-prev-month:hover,
.flatpickr-months .flatpickr-next-month:hover {
    background: rgba(255,255,255,0.1) !important;
    border-radius: 6px !important;
}
.flatpickr-weekdays {
    background: #f8fafc !important;
}
.flatpickr-weekday {
    color: #64748b !important;
    font-weight: 600 !important;
    font-size: 12px !important;
}
.flatpickr-day {
    border-radius: 8px !important;
    font-weight: 500 !important;
    color: #1f2937 !important;
}
.flatpickr-day:hover {
    background: #eff6ff !important;
    border-color: #eff6ff !important;
}
.flatpickr-day.today {
    border-color: #3b82f6 !important;
    background: transparent !important;
}
.flatpickr-day.selected {
    background: #3b82f6 !important;
    border-color: #3b82f6 !important;
    color: #fff !important;
}
.flatpickr-day.selected:hover {
    background: #2563eb !important;
    border-color: #2563eb !important;
}
.flatpickr-day.flatpickr-disabled {
    color: #d1d5db !important;
}
.flatpickr-innerContainer {
    padding: 8px !important;
}
/* Buttons */
.btn {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    gap: 6px;
    padding: 8px 16px;
    font-size: 14px;
    font-weight: 500;
    border-radius: 8px;
    border: none;
    cursor: pointer;
    transition: all 0.2s;
    text-decoration: none;
}
.btn .dashicons {
    font-size: 16px;
    width: 16px;
    height: 16px;
}
.btn-primary { background: #3b82f6; color: #fff; }
.btn-primary:hover { background: #2563eb; color: #fff; }
.btn-secondary { background: #f3f4f6; color: #374151; border: 1px solid #d1d5db; }
.btn-secondary:hover { background: #e5e7eb; }
.btn-success { background: #10b981; color: #fff; }
.btn-success:hover { background: #059669; color: #fff; }
.btn-warning { background: #f59e0b; color: #fff; }
.btn-warning:hover { background: #d97706; color: #fff; }
.btn-danger { background: transparent; color: #ef4444; border: 1px solid #ef4444; }
.btn-danger:hover { background: #ef4444; color: #fff; }
.btn-sm { padding: 6px 12px; font-size: 13px; }
/* Data Table - Premium Design */
.data-table-wrapper {
    background: #fff;
    border: 1px solid #e5e7eb;
    border-radius: 16px;
    overflow: hidden;
    box-shadow: 0 1px 3px rgba(0,0,0,0.04);
}
.data-table {
    width: 100%;
    border-collapse: collapse;
}
.data-table thead { background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%); }
.data-table th {
    padding: 14px 20px;
    text-align: left;
    font-size: 11px;
    font-weight: 700;
    color: #64748b;
    text-transform: uppercase;
    letter-spacing: 0.8px;
    border-bottom: 2px solid #e2e8f0;
}
.data-table td {
    padding: 16px 20px;
    font-size: 14px;
    color: #374151;
    border-bottom: 1px solid #f1f5f9;
    vertical-align: middle;
}
.data-table tbody tr {
    transition: all 0.15s ease;
}
.data-table tbody tr:hover { 
    background: linear-gradient(135deg, #f8fafc 0%, #eff6ff 100%);
}
.data-table tbody tr:last-child td {
    border-bottom: none;
}
/* User Info - Premium Card Style */
.user-info { 
    display: flex; 
    align-items: center;
    gap: 12px; 
}
.user-avatar {
    width: 42px;
    height: 42px;
    background: linear-gradient(135deg, #3b82f6 0%, #8b5cf6 100%);
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    color: #fff;
    font-weight: 700;
    font-size: 15px;
    text-transform: uppercase;
    flex-shrink: 0;
}
.user-details {
    display: flex;
    flex-direction: column;
    gap: 3px;
}
.user-name { 
    font-weight: 600; 
    color: #1e293b;
    font-size: 14px;
}
.user-phone { 
    font-size: 13px; 
    color: #64748b;
    display: flex;
    align-items: center;
    gap: 4px;
}
.user-phone::before {
    content: '📱';
    font-size: 11px;
}
/* DateTime Info - Premium Style */
.datetime-info { 
    display: flex; 
    align-items: center;
    gap: 10px; 
}
.datetime-box {
    display: flex;
    flex-direction: column;
    align-items: center;
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
    border: 1px solid #e2e8f0;
    border-radius: 10px;
    padding: 8px 12px;
    min-width: 70px;
}
.datetime-box.highlight {
    background: linear-gradient(135deg, #eff6ff 0%, #dbeafe 100%);
    border-color: #93c5fd;
}
.datetime-day {
    font-size: 18px;
    font-weight: 700;
    color: #1e293b;
    line-height: 1;
}
.datetime-month {
    font-size: 11px;
    font-weight: 600;
    color: #64748b;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}
.datetime-year {
    font-size: 10px;
    color: #94a3b8;
}
.datetime-time {
    display: flex;
    align-items: center;
    gap: 6px;
    background: linear-gradient(135deg, #f0fdf4 0%, #dcfce7 100%);
    border: 1px solid #86efac;
    border-radius: 8px;
    padding: 6px 10px;
    font-size: 14px;
    font-weight: 600;
    color: #166534;
}
.datetime-time .dashicons {
    font-size: 14px;
    width: 14px;
    height: 14px;
    color: #22c55e;
}
/* Created At - Subtle Style */
.created-info {
    display: flex;
    flex-direction: column;
    gap: 2px;
    color: #64748b;
    font-size: 13px;
}
.created-date {
    font-weight: 500;
    color: #475569;
}
.created-time {
    font-size: 12px;
    color: #94a3b8;
}
.table-actions { display: flex; gap: 8px; align-items: center; flex-wrap: wrap; }
/* Status Badges - Premium */
.status-badge {
    display: inline-flex;
    align-items: center;
    gap: 6px;
    padding: 6px 14px;
    font-size: 12px;
    font-weight: 600;
    border-radius: 20px;
    text-transform: uppercase;
    letter-spacing: 0.3px;
}
.status-badge::before {
    content: '';
    width: 6px;
    height: 6px;
    border-radius: 50%;
}
.status-pending { 
    background: linear-gradient(135deg, #fffbeb 0%, #fef3c7 100%); 
    color: #92400e;
    border: 1px solid #fcd34d;
}
.status-pending::before { background: #f59e0b; }
.status-confirmed { 
    background: linear-gradient(135deg, #ecfdf5 0%, #d1fae5 100%); 
    color: #065f46;
    border: 1px solid #6ee7b7;
}
.status-confirmed::before { background: #10b981; }
.status-rejected { 
    background: linear-gradient(135deg, #fef2f2 0%, #fee2e2 100%); 
    color: #991b1b;
    border: 1px solid #fca5a5;
}
.status-rejected::before { background: #ef4444; }
.status-cancelled { 
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%); 
    color: #475569;
    border: 1px solid #cbd5e1;
}
.status-cancelled::before { background: #94a3b8; }
/* Empty State */
.empty-state {
    text-align: center;
    padding: 48px 24px;
    color: #6b7280;
}
.empty-state .dashicons {
    font-size: 48px;
    width: 48px;
    height: 48px;
    color: #d1d5db;
    margin-bottom: 12px;
}
/* Pagination */
.pagination-wrapper {
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 16px 20px;
    background: #f9fafb;
    border-top: 1px solid #e5e7eb;
}
.pagination-info {
    font-size: 14px;
    color: #6b7280;
}
.pagination-links {
    display: flex;
    gap: 4px;
}
.pagination-links a,
.pagination-links span {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    min-width: 36px;
    height: 36px;
    padding: 0 12px;
    font-size: 14px;
    font-weight: 500;
    border-radius: 8px;
    text-decoration: none;
    transition: all 0.15s;
}
.pagination-links a {
    background: #fff;
    color: #374151;
    border: 1px solid #d1d5db;
}
.pagination-links a:hover {
    background: #3b82f6;
    color: #fff;
    border-color: #3b82f6;
}
.pagination-links .current {
    background: #3b82f6;
    color: #fff;
    border: 1px solid #3b82f6;
}
.pagination-links .dots {
    background: transparent;
    border: none;
    color: #9ca3af;
}
/* Modal */
.modal-overlay {
    display: none;
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    z-index: 100000;
    align-items: center;
    justify-content: center;
}
.modal-overlay.active {
    display: flex;
}
.modal-box {
    background: #fff;
    border-radius: 16px;
    box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.25);
    max-width: 400px;
    width: 90%;
    overflow: hidden;
    animation: modalIn 0.2s ease;
}
@keyframes modalIn {
    from { opacity: 0; transform: scale(0.95) translateY(-10px); }
    to { opacity: 1; transform: scale(1) translateY(0); }
}
.modal-header {
    display: flex;
    align-items: center;
    gap: 12px;
    padding: 20px 24px;
    border-bottom: 1px solid #e5e7eb;
}
.modal-icon {
    width: 48px;
    height: 48px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
}
.modal-icon .dashicons {
    font-size: 24px;
    width: 24px;
    height: 24px;
}
.modal-icon.success { background: #ecfdf5; color: #10b981; }
.modal-icon.warning { background: #fffbeb; color: #f59e0b; }
.modal-icon.danger { background: #fef2f2; color: #ef4444; }
.modal-title {
    font-size: 18px;
    font-weight: 600;
    color: #111827;
    margin: 0;
}
.modal-body {
    padding: 20px 24px;
}
.modal-body p {
    margin: 0;
    font-size: 14px;
    color: #6b7280;
    line-height: 1.6;
}
.modal-footer {
    display: flex;
    gap: 12px;
    padding: 16px 24px;
    background: #f9fafb;
    border-top: 1px solid #e5e7eb;
}
.modal-footer .btn {
    flex: 1;
}
/* Modal Details */
.modal-details {
    background: #f9fafb;
    border-radius: 8px;
    padding: 16px;
}
.detail-row {
    display: flex;
    justify-content: space-between;
    padding: 8px 0;
    border-bottom: 1px solid #e5e7eb;
}
.detail-row:last-child {
    border-bottom: none;
}
.detail-label {
    font-size: 13px;
    color: #6b7280;
    font-weight: 500;
}
.detail-value {
    font-size: 14px;
    color: #111827;
    font-weight: 600;
}
</style>

<div class="appointment-admin-wrap">
    <!-- Page Header -->
    <div class="admin-page-header">
        <h1 class="admin-page-title">
            <span class="dashicons dashicons-calendar-alt"></span>
            <?php esc_html_e('Randevular', 'appointment-booking'); ?>
        </h1>
        <button type="button" class="btn-add-appointment" onclick="openAddAppointmentModal()">
            <span class="dashicons dashicons-plus-alt"></span>
            Yeni Randevu Ekle
        </button>
    </div>
    
    <?php if (isset($_GET['added'])): ?>
    <div style="background: linear-gradient(135deg, #ecfdf5 0%, #d1fae5 100%); border: 2px solid #6ee7b7; color: #065f46; padding: 16px 20px; border-radius: 12px; margin-bottom: 20px; display: flex; align-items: center; gap: 10px; font-weight: 600; animation: slideDown 0.3s;">
        <span style="font-size: 20px;">✓</span> Randevu başarıyla eklendi!
    </div>
    <?php endif; ?>
    
    <?php if (isset($_GET['error'])): ?>
        <?php 
        $error_messages = array(
            'missing_fields' => 'Lütfen tüm zorunlu alanları doldurun.',
            'date_blocked' => 'Bu tarih veya saat blokeli. Lütfen başka bir zaman seçin.',
            'time_taken' => 'Bu tarih ve saatte zaten bir randevu var. Lütfen başka bir saat seçin.',
            'db_error' => 'Veritabanı hatası oluştu. Lütfen tekrar deneyin.'
        );
        $error_key = sanitize_text_field($_GET['error']);
        $error_msg = isset($error_messages[$error_key]) ? $error_messages[$error_key] : 'Bir hata oluştu.';
        ?>
        <div style="background: linear-gradient(135deg, #fef2f2 0%, #fee2e2 100%); border: 2px solid #fca5a5; color: #991b1b; padding: 16px 20px; border-radius: 12px; margin-bottom: 20px; display: flex; align-items: center; gap: 10px; font-weight: 600; animation: slideDown 0.3s;">
            <span style="font-size: 20px;">⚠</span> <?php echo esc_html($error_msg); ?>
        </div>
    <?php endif; ?>
    
    <!-- Statistics -->
    <div class="stats-grid">
        <div class="stat-card total">
            <div class="stat-icon">
                <span class="dashicons dashicons-list-view"></span>
            </div>
            <div class="stat-content">
                <div class="stat-number"><?php echo esc_html($stats['total']); ?></div>
                <div class="stat-label"><?php esc_html_e('Toplam', 'appointment-booking'); ?></div>
            </div>
        </div>
        <div class="stat-card pending">
            <div class="stat-icon">
                <span class="dashicons dashicons-clock"></span>
            </div>
            <div class="stat-content">
                <div class="stat-number"><?php echo esc_html($stats['pending']); ?></div>
                <div class="stat-label"><?php esc_html_e('Bekleyen', 'appointment-booking'); ?></div>
            </div>
        </div>
        <div class="stat-card confirmed">
            <div class="stat-icon">
                <span class="dashicons dashicons-yes-alt"></span>
            </div>
            <div class="stat-content">
                <div class="stat-number"><?php echo esc_html($stats['confirmed']); ?></div>
                <div class="stat-label"><?php esc_html_e('Onaylanan', 'appointment-booking'); ?></div>
            </div>
        </div>
        <div class="stat-card rejected">
            <div class="stat-icon">
                <span class="dashicons dashicons-dismiss"></span>
            </div>
            <div class="stat-content">
                <div class="stat-number"><?php echo esc_html($stats['rejected']); ?></div>
                <div class="stat-label"><?php esc_html_e('Reddedilen', 'appointment-booking'); ?></div>
            </div>
        </div>
    </div>
    
    <!-- Filters -->
    <div class="filter-bar">
        <form method="get" action="" class="filter-form">
            <input type="hidden" name="page" value="appointment-booking">
            
            <div class="filter-row">
                <div class="filter-group">
                    <div class="filter-icon-wrapper">
                        <span class="dashicons dashicons-filter"></span>
                    </div>
                    <select name="status" class="filter-select-styled">
                        <option value=""><?php esc_html_e('Tüm Durumlar', 'appointment-booking'); ?></option>
                        <option value="pending" <?php selected($status, 'pending'); ?>>⏳ <?php esc_html_e('Bekleyen', 'appointment-booking'); ?></option>
                        <option value="confirmed" <?php selected($status, 'confirmed'); ?>>✅ <?php esc_html_e('Onaylanan', 'appointment-booking'); ?></option>
                        <option value="rejected" <?php selected($status, 'rejected'); ?>>❌ <?php esc_html_e('Reddedilen', 'appointment-booking'); ?></option>
                    </select>
                </div>
                
                <div class="filter-group date-group">
                    <span class="dashicons dashicons-calendar-alt"></span>
                    <input type="text" name="date_from" value="<?php echo esc_attr($date_from); ?>" class="filter-date datepicker" placeholder="Başlangıç" readonly>
                    <span class="date-sep">—</span>
                    <input type="text" name="date_to" value="<?php echo esc_attr($date_to); ?>" class="filter-date datepicker" placeholder="Bitiş" readonly>
                </div>
                
                <div class="filter-group search-group">
                    <span class="dashicons dashicons-search"></span>
                    <input type="text" name="search" value="<?php echo esc_attr($search); ?>" placeholder="<?php esc_attr_e('Ad, soyad veya telefon ara...', 'appointment-booking'); ?>" class="filter-search-input">
                </div>
                
                <div class="filter-actions">
                    <button type="submit" class="btn-filter">
                        <span class="dashicons dashicons-search"></span>
                        <?php esc_html_e('Filtrele', 'appointment-booking'); ?>
                    </button>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=appointment-booking')); ?>" class="btn-clear">
                        <span class="dashicons dashicons-dismiss"></span>
                    </a>
                </div>
            </div>
        </form>
    </div>
    
    <!-- Appointments Table -->
    <div class="data-table-wrapper">
        <table class="data-table">
            <thead>
                <tr>
                    <th><?php esc_html_e('ID', 'appointment-booking'); ?></th>
                    <th><?php esc_html_e('Müşteri', 'appointment-booking'); ?></th>
                    <th><?php esc_html_e('Randevu', 'appointment-booking'); ?></th>
                    <th><?php esc_html_e('Durum', 'appointment-booking'); ?></th>
                    <th><?php esc_html_e('Oluşturulma', 'appointment-booking'); ?></th>
                    <th><?php esc_html_e('İşlemler', 'appointment-booking'); ?></th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($appointments)): ?>
                <tr>
                    <td colspan="6">
                        <div class="empty-state">
                            <span class="dashicons dashicons-calendar-alt"></span>
                            <p><?php esc_html_e('Randevu bulunamadı.', 'appointment-booking'); ?></p>
                        </div>
                    </td>
                </tr>
                <?php else: ?>
                <?php foreach ($appointments as $appointment): 
                    $initials = strtoupper(substr($appointment->user_name, 0, 1) . substr($appointment->user_surname, 0, 1));
                    $months_tr = array('', 'Oca', 'Şub', 'Mar', 'Nis', 'May', 'Haz', 'Tem', 'Ağu', 'Eyl', 'Eki', 'Kas', 'Ara');
                ?>
                <tr data-id="<?php echo esc_attr($appointment->id); ?>">
                    <td><span style="color: #94a3b8; font-weight: 500;">#<?php echo esc_html($appointment->id); ?></span></td>
                    <td>
                        <div class="user-info">
                            <div class="user-avatar"><?php echo esc_html($initials); ?></div>
                            <div class="user-details">
                                <span class="user-name"><?php echo esc_html($appointment->user_name . ' ' . $appointment->user_surname); ?></span>
                                <span class="user-phone"><?php echo esc_html(appointment_format_phone($appointment->user_phone)); ?></span>
                            </div>
                        </div>
                    </td>
                    <td>
                        <div class="datetime-info">
                            <div class="datetime-box highlight">
                                <span class="datetime-day"><?php echo esc_html(date('d', strtotime($appointment->appointment_date))); ?></span>
                                <span class="datetime-month"><?php echo esc_html($months_tr[date('n', strtotime($appointment->appointment_date))]); ?></span>
                            </div>
                            <div class="datetime-time">
                                <span class="dashicons dashicons-clock"></span>
                                <?php echo esc_html(date('H:i', strtotime($appointment->appointment_time))); ?>
                            </div>
                        </div>
                    </td>
                    <td>
                        <span class="status-badge status-<?php echo esc_attr($appointment->status); ?>">
                            <?php
                            $status_labels = array(
                                'pending' => __('Bekleyen', 'appointment-booking'),
                                'confirmed' => __('Onaylanan', 'appointment-booking'),
                                'rejected' => __('Reddedilen', 'appointment-booking'),
                                'cancelled' => __('İptal', 'appointment-booking')
                            );
                            echo esc_html($status_labels[$appointment->status]);
                            ?>
                        </span>
                    </td>
                    <td>
                        <div class="created-info">
                            <span class="created-date"><?php echo esc_html(date('d.m.Y', strtotime($appointment->created_at))); ?></span>
                            <span class="created-time"><?php echo esc_html(date('H:i', strtotime($appointment->created_at))); ?></span>
                        </div>
                    </td>
                    <td>
                        <div class="table-actions">
                            <?php if ($appointment->status === 'pending'): ?>
                            <button class="btn btn-success btn-sm appointment-action" 
                                data-action="confirm" 
                                data-id="<?php echo esc_attr($appointment->id); ?>" 
                                data-name="<?php echo esc_attr($appointment->user_name . ' ' . $appointment->user_surname); ?>"
                                data-phone="<?php echo esc_attr(appointment_format_phone($appointment->user_phone)); ?>"
                                data-date="<?php echo esc_attr(date('d.m.Y', strtotime($appointment->appointment_date))); ?>"
                                data-time="<?php echo esc_attr(date('H:i', strtotime($appointment->appointment_time))); ?>">
                                <span class="dashicons dashicons-yes"></span>
                                <?php esc_html_e('Onayla', 'appointment-booking'); ?>
                            </button>
                            <button class="btn btn-warning btn-sm appointment-action" 
                                data-action="reject" 
                                data-id="<?php echo esc_attr($appointment->id); ?>" 
                                data-name="<?php echo esc_attr($appointment->user_name . ' ' . $appointment->user_surname); ?>"
                                data-phone="<?php echo esc_attr(appointment_format_phone($appointment->user_phone)); ?>"
                                data-date="<?php echo esc_attr(date('d.m.Y', strtotime($appointment->appointment_date))); ?>"
                                data-time="<?php echo esc_attr(date('H:i', strtotime($appointment->appointment_time))); ?>">
                                <span class="dashicons dashicons-no"></span>
                            </button>
                            <?php elseif ($appointment->status === 'confirmed' || $appointment->status === 'rejected'): ?>
                            <button class="btn btn-secondary btn-sm appointment-action" 
                                data-action="revert" 
                                data-id="<?php echo esc_attr($appointment->id); ?>" 
                                data-name="<?php echo esc_attr($appointment->user_name . ' ' . $appointment->user_surname); ?>"
                                data-phone="<?php echo esc_attr(appointment_format_phone($appointment->user_phone)); ?>"
                                data-date="<?php echo esc_attr(date('d.m.Y', strtotime($appointment->appointment_date))); ?>"
                                data-time="<?php echo esc_attr(date('H:i', strtotime($appointment->appointment_time))); ?>"
                                title="Bekleyene Geri Al">
                                <span class="dashicons dashicons-undo"></span>
                            </button>
                            <?php endif; ?>
                            <button class="btn btn-danger btn-sm appointment-action" 
                                data-action="delete" 
                                data-id="<?php echo esc_attr($appointment->id); ?>" 
                                data-name="<?php echo esc_attr($appointment->user_name . ' ' . $appointment->user_surname); ?>"
                                data-phone="<?php echo esc_attr(appointment_format_phone($appointment->user_phone)); ?>"
                                data-date="<?php echo esc_attr(date('d.m.Y', strtotime($appointment->appointment_date))); ?>"
                                data-time="<?php echo esc_attr(date('H:i', strtotime($appointment->appointment_time))); ?>">
                                <span class="dashicons dashicons-trash"></span>
                            </button>
                        </div>
                    </td>
                </tr>
                <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
        
        <?php if ($pagination['total_pages'] > 1): ?>
        <!-- Pagination -->
        <div class="pagination-wrapper">
            <div class="pagination-info">
                <?php
                $start = (($pagination['current_page'] - 1) * $pagination['per_page']) + 1;
                $end = min($pagination['current_page'] * $pagination['per_page'], $pagination['total_items']);
                printf(
                    esc_html__('%d-%d / %d kayıt gösteriliyor', 'appointment-booking'),
                    $start,
                    $end,
                    $pagination['total_items']
                );
                ?>
            </div>
            <div class="pagination-links">
                <?php
                $base_url = admin_url('admin.php?page=appointment-booking');
                if ($status) $base_url .= '&status=' . urlencode($status);
                if ($date_from) $base_url .= '&date_from=' . urlencode($date_from);
                if ($date_to) $base_url .= '&date_to=' . urlencode($date_to);
                if ($search) $base_url .= '&search=' . urlencode($search);
                
                // Previous
                if ($pagination['current_page'] > 1): ?>
                    <a href="<?php echo esc_url($base_url . '&paged=' . ($pagination['current_page'] - 1)); ?>">‹</a>
                <?php endif;
                
                // Page numbers
                $range = 2;
                for ($i = 1; $i <= $pagination['total_pages']; $i++):
                    if ($i == 1 || $i == $pagination['total_pages'] || ($i >= $pagination['current_page'] - $range && $i <= $pagination['current_page'] + $range)):
                        if ($i == $pagination['current_page']): ?>
                            <span class="current"><?php echo $i; ?></span>
                        <?php else: ?>
                            <a href="<?php echo esc_url($base_url . '&paged=' . $i); ?>"><?php echo $i; ?></a>
                        <?php endif;
                    elseif ($i == $pagination['current_page'] - $range - 1 || $i == $pagination['current_page'] + $range + 1): ?>
                        <span class="dots">...</span>
                    <?php endif;
                endfor;
                
                // Next
                if ($pagination['current_page'] < $pagination['total_pages']): ?>
                    <a href="<?php echo esc_url($base_url . '&paged=' . ($pagination['current_page'] + 1)); ?>">›</a>
                <?php endif; ?>
            </div>
        </div>
        <?php endif; ?>
    </div>
</div>

<!-- Modal -->
<div class="modal-overlay" id="actionModal">
    <div class="modal-box">
        <div class="modal-header">
            <div class="modal-icon" id="modalIcon">
                <span class="dashicons"></span>
            </div>
            <h3 class="modal-title" id="modalTitle"></h3>
        </div>
        <div class="modal-body">
            <div class="modal-details" id="modalDetails">
                <div class="detail-row">
                    <span class="detail-label">Ad Soyad:</span>
                    <span class="detail-value" id="detailName"></span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Telefon:</span>
                    <span class="detail-value" id="detailPhone"></span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Tarih:</span>
                    <span class="detail-value" id="detailDate"></span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Saat:</span>
                    <span class="detail-value" id="detailTime"></span>
                </div>
            </div>
            <p id="modalMessage" style="margin-top: 16px; padding-top: 16px; border-top: 1px solid #e5e7eb;"></p>
        </div>
        <div class="modal-footer">
            <button type="button" class="btn btn-secondary" id="modalCancel"><?php esc_html_e('İptal', 'appointment-booking'); ?></button>
            <button type="button" class="btn" id="modalConfirm"><?php esc_html_e('Onayla', 'appointment-booking'); ?></button>
        </div>
    </div>
</div>

<script>
jQuery(document).ready(function($) {
    var currentAction = null;
    var currentId = null;
    var currentRow = null;
    
    // Initialize Flatpickr for date inputs
    if (typeof flatpickr !== 'undefined') {
        flatpickr.localize(flatpickr.l10ns.tr);
        
        $('.datepicker').flatpickr({
            dateFormat: 'Y-m-d',
            altInput: true,
            altFormat: 'd M Y',
            locale: 'tr',
            disableMobile: true,
            allowInput: false,
            clickOpens: true,
            theme: 'light'
        });
    }
    
    // Format phone number helper
    function formatPhone(phone) {
        var digits = phone.replace(/\D/g, '');
        if (digits.length === 10 && digits[0] !== '0') {
            digits = '0' + digits;
        }
        if (digits.length === 11) {
            return digits.slice(0, 4) + ' ' + digits.slice(4, 7) + ' ' + digits.slice(7, 9) + ' ' + digits.slice(9, 11);
        }
        return phone;
    }
    
    // Modal config
    var modalConfig = {
        confirm: {
            icon: 'dashicons-yes-alt',
            iconClass: 'success',
            title: 'Randevuyu Onayla',
            message: 'Bu randevuyu onaylamak istediğinize emin misiniz?',
            btnClass: 'btn-success',
            btnText: 'Onayla'
        },
        reject: {
            icon: 'dashicons-warning',
            iconClass: 'warning',
            title: 'Randevuyu Reddet',
            message: 'Bu randevuyu reddetmek istediğinize emin misiniz?',
            btnClass: 'btn-warning',
            btnText: 'Reddet'
        },
        revert: {
            icon: 'dashicons-undo',
            iconClass: 'warning',
            title: 'Bekleyene Geri Al',
            message: 'Bu randevuyu bekleyen durumuna geri almak istediğinize emin misiniz?',
            btnClass: 'btn-secondary',
            btnText: 'Geri Al'
        },
        delete: {
            icon: 'dashicons-trash',
            iconClass: 'danger',
            title: 'Randevuyu Sil',
            message: 'Bu randevuyu silmek istediğinize emin misiniz? Bu işlem geri alınamaz.',
            btnClass: 'btn-danger',
            btnText: 'Sil'
        }
    };
    
    // Open modal
    $('.appointment-action').on('click', function() {
        currentAction = $(this).data('action');
        currentId = $(this).data('id');
        currentRow = $(this).closest('tr');
        
        var name = $(this).data('name');
        var phone = $(this).data('phone');
        var date = $(this).data('date');
        var time = $(this).data('time');
        
        var config = modalConfig[currentAction];
        
        $('#modalIcon').removeClass('success warning danger').addClass(config.iconClass);
        $('#modalIcon .dashicons').attr('class', 'dashicons ' + config.icon);
        $('#modalTitle').text(config.title);
        $('#modalMessage').text(config.message);
        $('#modalConfirm').removeClass('btn-success btn-warning btn-danger').addClass(config.btnClass).text(config.btnText);
        
        // Fill details with formatted phone
        $('#detailName').text(name);
        $('#detailPhone').text(formatPhone(phone));
        $('#detailDate').text(date);
        $('#detailTime').text(time);
        
        $('#actionModal').addClass('active');
    });
    
    // Close modal
    $('#modalCancel, .modal-overlay').on('click', function(e) {
        if (e.target === this) {
            $('#actionModal').removeClass('active');
            currentAction = null;
            currentId = null;
            currentRow = null;
        }
    });
    
    // Confirm action
    $('#modalConfirm').on('click', function() {
        if (!currentAction || !currentId) return;
        
        var $btn = $(this);
        $btn.prop('disabled', true).text('İşleniyor...');
        
        if (currentAction === 'delete') {
            $.post(ajaxurl, {
                action: 'appointment_admin_delete',
                nonce: '<?php echo wp_create_nonce("appointment_admin_nonce"); ?>',
                id: currentId
            }, function(r) {
                if (r.success) {
                    currentRow.fadeOut(300, function() { $(this).remove(); });
                }
                closeModal();
            });
        } else {
            var newStatus = currentAction === 'confirm' ? 'confirmed' : (currentAction === 'revert' ? 'pending' : 'rejected');
            $.post(ajaxurl, {
                action: 'appointment_admin_update_status',
                nonce: '<?php echo wp_create_nonce("appointment_admin_nonce"); ?>',
                id: currentId,
                status: newStatus
            }, function(r) {
                if (r.success) {
                    // Reload page to refresh buttons
                    location.reload();
                }
                closeModal();
            });
        }
    });
    
    function closeModal() {
        $('#actionModal').removeClass('active');
        $('#modalConfirm').prop('disabled', false);
        currentAction = null;
        currentId = null;
        currentRow = null;
    }
    
    // ESC key to close
    $(document).on('keydown', function(e) {
        if (e.key === 'Escape' && $('#actionModal').hasClass('active')) {
            closeModal();
        }
    });
});
</script>


<!-- Add Appointment Modal -->
<div id="addAppointmentModal" class="appointment-modal" style="display:none;">
    <div class="modal-overlay" onclick="closeAddAppointmentModal()"></div>
    <div class="modal-content">
        <div class="modal-header">
            <h3>Yeni Randevu Ekle</h3>
            <button class="modal-close" onclick="closeAddAppointmentModal()">&times;</button>
        </div>
        <div class="modal-body">
            <div id="addAppointmentMessage" style="display:none; padding: 12px; border-radius: 8px; margin-bottom: 16px; font-size: 14px; font-weight: 600;"></div>
            
            <form id="addAppointmentForm">
                <div class="form-row">
                    <div class="form-group">
                        <label>Ad <span style="color:#dc2626;">*</span></label>
                        <input type="text" name="user_name" required>
                    </div>
                    
                    <div class="form-group">
                        <label>Soyad <span style="color:#dc2626;">*</span></label>
                        <input type="text" name="user_surname" required>
                    </div>
                </div>
                
                <div class="form-group">
                    <label>Telefon <span style="color:#dc2626;">*</span></label>
                    <input type="tel" name="user_phone" id="adminPhoneInput" placeholder="(5XX) XXX XX XX" maxlength="15" required>
                </div>
                
                <div class="form-group">
                    <label>Randevu Tarihi <span style="color:#dc2626;">*</span></label>
                    <input type="text" name="appointment_date" id="adminDatePicker" placeholder="Tarih seçin" required readonly>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label>Başlangıç <span style="color:#dc2626;">*</span></label>
                        <input type="time" name="appointment_time" id="appointmentStartTime" required>
                    </div>
                    
                    <div class="form-group">
                        <label>Bitiş <span style="color:#dc2626;">*</span></label>
                        <input type="time" name="appointment_end_time" id="appointmentEndTime" required>
                    </div>
                </div>
                
                <div class="form-group">
                    <label>Durum</label>
                    <select name="status">
                        <option value="pending">Bekliyor</option>
                        <option value="confirmed">Onaylandı</option>
                        <option value="rejected">İptal Edildi</option>
                    </select>
                </div>
                
                <div class="modal-actions">
                    <button type="button" class="btn-cancel" onclick="closeAddAppointmentModal()">İptal</button>
                    <button type="submit" class="btn-save" id="btnSaveAppointment">Randevu Ekle</button>
                </div>
            </form>
        </div>
    </div>
</div>

<style>
.appointment-modal { position: fixed; top: 0; left: 0; right: 0; bottom: 0; z-index: 100000; display: flex; align-items: center; justify-content: center; animation: fadeIn 0.2s; }
@keyframes fadeIn { from { opacity: 0; } to { opacity: 1; } }
.modal-overlay { position: absolute; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.7); backdrop-filter: blur(8px); -webkit-backdrop-filter: blur(8px); }
.modal-content { position: relative; background: #fff; border-radius: 16px; box-shadow: 0 20px 60px rgba(0,0,0,0.3); max-width: 420px; width: 92%; max-height: 85vh; overflow-y: auto; animation: slideUp 0.3s; z-index: 100001; }
@keyframes slideUp { from { transform: translateY(30px); opacity: 0; } to { transform: translateY(0); opacity: 1; } }
.modal-header { padding: 18px 20px; border-bottom: 2px solid #f3f4f6; display: flex; align-items: center; justify-content: space-between; background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%); border-radius: 16px 16px 0 0; }
.modal-header h3 { margin: 0; font-size: 17px; font-weight: 700; color: #111827; display: flex; align-items: center; gap: 8px; }
.modal-header h3:before { content: '📝'; font-size: 20px; }
.modal-close { width: 32px; height: 32px; display: flex; align-items: center; justify-content: center; background: #fff; border: 2px solid #e5e7eb; border-radius: 8px; font-size: 22px; color: #6b7280; cursor: pointer; transition: all 0.2s; line-height: 1; }
.modal-close:hover { background: #fee2e2; border-color: #fecaca; color: #dc2626; transform: rotate(90deg); }
.modal-body { padding: 20px; }
.form-row { display: grid; grid-template-columns: 1fr 1fr; gap: 12px; }
.form-group { margin-bottom: 16px; }
.form-group label { display: block; margin-bottom: 6px; font-size: 13px; font-weight: 600; color: #374151; }
.form-group input, .form-group select { width: 100%; padding: 10px 12px; border: 2px solid #e5e7eb; border-radius: 8px; font-size: 14px; transition: all 0.2s; background: #f9fafb; box-sizing: border-box; }
.form-group input:focus, .form-group select:focus { outline: none; border-color: #10b981; background: #fff; box-shadow: 0 0 0 3px rgba(16,185,129,0.1); }
.modal-actions { display: flex; gap: 10px; margin-top: 20px; padding-top: 16px; border-top: 2px solid #f3f4f6; }
.btn-cancel { flex: 1; padding: 11px; background: #f3f4f6; color: #374151; border: none; border-radius: 8px; font-size: 14px; font-weight: 600; cursor: pointer; transition: all 0.2s; }
.btn-cancel:hover { background: #e5e7eb; }
.btn-save { flex: 1; padding: 11px; background: linear-gradient(135deg, #10b981 0%, #059669 100%); color: #fff; border: none; border-radius: 8px; font-size: 14px; font-weight: 600; cursor: pointer; transition: all 0.2s; box-shadow: 0 4px 12px rgba(16,185,129,0.3); }
.btn-save:hover { transform: translateY(-2px); box-shadow: 0 6px 16px rgba(16,185,129,0.4); }
@media (max-width: 500px) {
    .form-row { grid-template-columns: 1fr; }
    .modal-content { max-width: 95%; }
}
/* Flatpickr z-index fix */
.flatpickr-calendar { z-index: 100002 !important; }
</style>

<script>
function openAddAppointmentModal() {
    document.getElementById('addAppointmentModal').style.display = 'flex';
    document.getElementById('addAppointmentMessage').style.display = 'none';
}

function closeAddAppointmentModal() {
    document.getElementById('addAppointmentModal').style.display = 'none';
    document.getElementById('addAppointmentForm').reset();
    document.getElementById('addAppointmentMessage').style.display = 'none';
}

// Phone formatting
document.addEventListener('DOMContentLoaded', function() {
    var phoneInput = document.getElementById('adminPhoneInput');
    if (phoneInput) {
        phoneInput.addEventListener('input', function(e) {
            var value = e.target.value.replace(/\D/g, '');
            if (value.length > 10) value = value.slice(0, 10);
            
            var formatted = '';
            if (value.length > 0) {
                formatted = '(' + value.slice(0, 3);
                if (value.length > 3) {
                    formatted += ') ' + value.slice(3, 6);
                    if (value.length > 6) {
                        formatted += ' ' + value.slice(6, 8);
                        if (value.length > 8) {
                            formatted += ' ' + value.slice(8, 10);
                        }
                    }
                }
            }
            e.target.value = formatted;
        });
    }
    
    // Initialize flatpickr for admin date picker
    if (typeof flatpickr !== 'undefined') {
        var adminDatePicker = document.getElementById('adminDatePicker');
        if (adminDatePicker) {
            flatpickr(adminDatePicker, {
                dateFormat: 'Y-m-d',
                altInput: true,
                altFormat: 'd.m.Y',
                locale: 'tr',
                disableMobile: true,
                allowInput: false,
                minDate: 'today',
                theme: 'light'
            });
        }
    }
    
    // Auto-fill end time when start time changes
    var startTimeInput = document.getElementById('appointmentStartTime');
    var endTimeInput = document.getElementById('appointmentEndTime');
    if (startTimeInput && endTimeInput) {
        startTimeInput.addEventListener('change', function() {
            if (this.value) {
                var parts = this.value.split(':');
                var hours = parseInt(parts[0]);
                var minutes = parseInt(parts[1]);
                
                // Add 1 hour
                minutes += 60;
                if (minutes >= 60) {
                    hours += Math.floor(minutes / 60);
                    minutes = minutes % 60;
                }
                if (hours >= 24) hours = hours % 24;
                
                var endTime = String(hours).padStart(2, '0') + ':' + String(minutes).padStart(2, '0');
                endTimeInput.value = endTime;
            }
        });
    }
    
    // AJAX form submission
    var addAppointmentForm = document.getElementById('addAppointmentForm');
    if (addAppointmentForm) {
        addAppointmentForm.addEventListener('submit', function(e) {
            e.preventDefault();
            
            var formData = new FormData(this);
            formData.append('action', 'appointment_admin_add');
            formData.append('nonce', '<?php echo wp_create_nonce("appointment_admin_nonce"); ?>');
            
            var submitBtn = document.getElementById('btnSaveAppointment');
            var messageDiv = document.getElementById('addAppointmentMessage');
            
            submitBtn.disabled = true;
            submitBtn.textContent = 'Kaydediliyor...';
            messageDiv.style.display = 'none';
            
            jQuery.ajax({
                url: ajaxurl,
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                success: function(response) {
                    if (response.success) {
                        messageDiv.style.display = 'block';
                        messageDiv.style.background = 'linear-gradient(135deg, #ecfdf5 0%, #d1fae5 100%)';
                        messageDiv.style.border = '2px solid #6ee7b7';
                        messageDiv.style.color = '#065f46';
                        messageDiv.innerHTML = '<span style="font-size: 18px;">✓</span> ' + response.data.message;
                        
                        addAppointmentForm.reset();
                        
                        setTimeout(function() {
                            location.reload();
                        }, 1500);
                    } else {
                        messageDiv.style.display = 'block';
                        messageDiv.style.background = 'linear-gradient(135deg, #fef2f2 0%, #fee2e2 100%)';
                        messageDiv.style.border = '2px solid #fca5a5';
                        messageDiv.style.color = '#991b1b';
                        messageDiv.innerHTML = '<span style="font-size: 18px;">⚠</span> ' + response.data.message;
                        
                        submitBtn.disabled = false;
                        submitBtn.textContent = 'Randevu Ekle';
                    }
                },
                error: function() {
                    messageDiv.style.display = 'block';
                    messageDiv.style.background = 'linear-gradient(135deg, #fef2f2 0%, #fee2e2 100%)';
                    messageDiv.style.border = '2px solid #fca5a5';
                    messageDiv.style.color = '#991b1b';
                    messageDiv.innerHTML = '<span style="font-size: 18px;">⚠</span> Bir hata oluştu. Lütfen tekrar deneyin.';
                    
                    submitBtn.disabled = false;
                    submitBtn.textContent = 'Randevu Ekle';
                }
            });
        });
    }
});
</script>
